/* builder-manager.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public errordomain Valide.BuilderError
{
  /**
   * Builder not found
   */
  NOT_FOUND
}

/**
 * The builder manager
 */
public class Valide.BuilderManager : Object
{
  private static BuilderManager instance;
  private List<string> builders;

  /**
   * The executable manager
   */
  public ExecutableManager executables
  {
    get;
    set;
  }

  private BuilderManager ()
  {
  }

  construct
  {
    this.builders = new List<string> ();
    this.register_builder (typeof (NativeBuilder), typeof (NativeBuilderOptions));
  }

  /**
   * Get the uniq instance of the builder manager
   *
   * @return The BuilderManager
   */
  public static BuilderManager get_instance ()
  {
    if (instance == null)
    {
      instance = new BuilderManager ();
    }
    return instance;
  }

  /**
   * Register a new builder
   *
   * @param The Type of the new builder
   * @param The type ofthe builder options
   */
  public void register_builder (Type builder, Type builder_options)
  {
    string builder_name;

    builder_name = builder.name ();
    if (builder_name.has_prefix ("Valide"))
    {
      builder_name = builder_name.substring (6);
    }
    if (builder_name.has_suffix ("Builder"))
    {
      builder_name = builder_name.substring (0, builder_name.length - 7);
    }
    if (this.builders.find_custom (builder_name, strcmp) == null)
    {
      this.builders.append (builder_name);
    }
  }

  /**
   * Create a new builder from its name
   *
   * @param project The project
   *
   * @return A new builder
   */
  public Builder create_builder (Project project) throws BuilderError
  {
    Type type;
    string type_name;
    Builder builder = null;

    if (project.builder == null)
    {
      project.builder = "native";
    }
    // Check the plugin builder
    type_name = project.builder.up (1) + project.builder.substring (1).down () + "Builder";
    type = Type.from_name (type_name);
    if (type == 0)
    {
      // Check the internal builder
      type_name = "Valide" + type_name;
      type = Type.from_name (type_name);
    }
    if (type != 0)
    {
      builder = Object.new (type,
                            "executables", this.executables,
                            "project", project) as Builder;
    }
    else
    {
      throw new BuilderError.NOT_FOUND (_("The builder '%s' doesn't exist!"),
                                        project.builder);
    }
    return builder;
  }

  /**
   * Get the list of the available builders
   *
   * @return The list of the available builders
   */
  public unowned List<string> get_builders ()
  {
    return builders;
  }
}

