/* compiler.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public errordomain Valide.ErrorParsing
{
  FILE,
  LINE
}

/**
 * The vala compiler
 */
public class Valide.Compiler : Executable
{
  /**
   * The project
   */
  public Project project
  {
    get;
    construct;
  }

  private static const string EXECUTABLE_PATH = "valac -q";
  /**
   * @see Valide.Executable.executable
   */
  public override string executable
  {
    get { return EXECUTABLE_PATH; }
    set {}
  }

  /**
   * Get the filename from command line
   *
   * @param line A comand line
   *
   * @return The name of the file
   */
  protected virtual string get_file (string line) throws Error
  {
    string[] fields;

    fields = line.split (":", -1);
    if (strv_length (fields) < 2)
    {
    	throw new ErrorParsing.FILE (line);
    }
    return fields[0];
  }

  /**
   * Get the line from command line
   *
   * @param line A comand line
   * @param pos the line position
   */
  protected virtual void get_line (string line, out Position pos) throws Error
  {
    Regex regex;
    MatchInfo match_info;

    pos = Position ();
    regex = new Regex ("([\\d]+)", //"([\\d]+)\\.([\\d]+)-([\\d]+)\\.([\\d]+)",
                       RegexCompileFlags.OPTIMIZE,
                       RegexMatchFlags.NOTEMPTY);

    regex.match (line, 0, out match_info);

    int i = 0;
    while (match_info.matches ())
    {
      switch (i)
      {
        case 0:
          pos.start_line = int.parse (match_info.fetch (0)) - 1;
        break;
        case 1:
          pos.start_row = int.parse (match_info.fetch (0)) - 1;
        break;
        case 2:
          pos.end_line = int.parse (match_info.fetch (0)) - 1;
        break;
        case 3:
          pos.end_row = int.parse (match_info.fetch (0));
        break;
      }
      i++;
      match_info.next ();
    }
    if (i < 4)
    {
      pos.end_line = pos.start_line;
      pos.start_row = pos.end_row = 0;
    }
  }

  /**
   * @see Valide.Executable.get_color_line
   */
  public override void get_color_line (ref string line, out Gdk.Color color)
  {
    if (line.has_prefix ("Compilation failed"))
    {
      Gdk.Color.parse (Utils.Color.ERROR, out color);
    }
    else if (line.index_of ("error:") != -1)
    {
      Gdk.Color.parse (Utils.Color.ERROR, out color);
    }
    else if (line.has_prefix ("Compilation succeeded"))
    {
      Gdk.Color.parse (Utils.Color.SUCCESS, out color);
    }
    else if (line.index_of ("warning:") != -1)
    {
      Gdk.Color.parse (Utils.Color.WARNING, out color);
    }
    else
    {
      base.get_color_line (ref line, out color);
    }
  }

  /**
   * @see Valide.Executable.line_activated
   */
  public override void line_activated (string line)
  {
    if (line != this.get_start_msg() && line != this.get_end_msg())
    {
      try
      {
        if (this.project != null)
        {
          string file;
          Document document;

          file = this.get_file (line);
          document = this.project.open_file (file);
          try
          {
            Position pos;

            this.get_line (line, out pos);
            document.select_range (pos);
            document.grab_focus ();
          }
          catch (Error e)
          {
            debug (e.message);
          }
        }
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
  }

  /**
   * Create a new Valide.Compiler
   *
   * @param project A project
   */
  public Compiler (Project project)
  {
    Object (project: project);
  }
}

