/* document-close-dialog.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/**
 * DocumentCloseDialog is a dialog box that displays the unsaved document when
 * close all or close the main window
 */
public class Valide.DocumentCloseDialog : AbstractCloseDialog
{
  private Gtk.ListStore list_store;

  /* enum for tree model columns */
  enum Col
  {
	  ACTIVE,
	  TITLE,
	  DOCUMENT,
    NB
  }

  private List<Document> _selected;
  /**
   * The list of the selected files
   */
  public List<Document> selected
  {
    get
    {
      Gtk.TreeIter iter;

      _selected = new List<Document> ();
      this.list_store.get_iter_first (out iter);
      do
      {
        bool active;
        Document document;

        this.list_store.get (iter, Col.ACTIVE, out active, Col.DOCUMENT, out document);
        if (active)
        {
          _selected.append (document);
        }
      } while (this.list_store.iter_next (ref iter));
      return _selected;
    }
  }

  private void active_toggled (Gtk.CellRendererToggle sender, string path_str)
  {
    bool active;
    Gtk.TreeIter iter;
    Gtk.TreePath path;

    path = new Gtk.TreePath.from_string (path_str);
    this.list_store.get_iter (out iter, path);
    this.list_store.get (iter, Col.ACTIVE, out active);

    active = !active;

    this.list_store.set (iter, Col.ACTIVE, active);
  }

  /**
   * Create a new Valide.DocumentCloseDialog
   *
   * @param documents The list of documents
   */
  public DocumentCloseDialog (List<Document> documents)
  {
    int unsave = 0;
    Gtk.CellRendererToggle toggle_render;
    Gtk.TreeViewColumn column;
    Gtk.CellRendererText text_render;

    foreach (Document document in documents)
    {
      if (!document.is_save)
      {
        unsave++;
      }
    }

    this.widgets.label.set_markup (_("<b>There are %d documents with unsaved changes. Save changes before closing?</b>").printf (unsave));

    this.list_store = new Gtk.ListStore (Col.NB, typeof (bool),
                                                 typeof (string),
                                                 typeof (Document));
    this.widgets.tree_view.model = this.list_store;

    toggle_render = new Gtk.CellRendererToggle ();
    toggle_render.toggled.connect (active_toggled);
    column = new Gtk.TreeViewColumn.with_attributes (_("Save"), toggle_render, "active", Col.ACTIVE, null);
    this.widgets.tree_view.append_column (column);

    text_render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("Filename"), text_render, "text", Col.TITLE, null);
    this.widgets.tree_view.append_column (column);

    foreach (Document document in documents)
    {
      if (!document.is_save)
      {
        Gtk.TreeIter iter;
        string path;

        path = document.title;
        if (path.has_prefix ("* "))
        {
          path = path.substring (2, path.length - 2);
        }
        this.list_store.append (out iter);
        this.list_store.set (iter, Col.ACTIVE, true, Col.TITLE, path, Col.DOCUMENT, document);
      }
    }
  }

  /**
   * @see Gtk.Window.set_transient_for
   */
  public void set_transient_for (Gtk.Window parent)
  {
    this.widgets.dialog.set_transient_for (parent);
  }

  /**
   * @see Gtk.Dialog.set_transient_for
   */
  public void set_modal (bool modal)
  {
    this.widgets.dialog.set_modal (modal);
  }

  /**
   * @see Gtk.Dialog.run
   */
  public int run ()
  {
    this.widgets.dialog.show_all ();
    return this.widgets.dialog.run ();
  }

  /**
   * @see Gtk.Widget.destroy
   */
  public override void destroy ()
  {
    this.widgets.dialog.destroy ();
    base.destroy ();
  }
}

