/* document-manager.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public errordomain Valide.DocumentError
{
  UNKNOW_DOCUMENT,
  BAD_URI,
  NO_DOCUMENT,
  MIME
}

/**
 * The manager for opened documents
 */
public class Valide.DocumentManager : Gtk.Notebook
{
  /**
   * The UI manager
   */
  public UIManager ui_manager
  {
    get;
    construct;
  }

  /**
   * New document is adding
   */
  public signal void tab_added (Document document);

  /**
   * Current document is saving
   */
  public signal void tab_saved (Document document);

  /**
   * Current document is closing
   */
  public signal void tab_removed (Document document);

  /**
   * The current document changed
   */
  public signal void tab_changed (Document document);

  /**
   * The state of the current document changed
   */
  public signal void tab_state_changed (Document document);

  /**
   * The tab_cursor_moved signal will be emitted when the user moved the caret
   */
  public signal void tab_cursor_moved (Document document, int row, int col);

  /**
   * The current opened document.
   */
  public Document current
  {
    get
    {
      return this.get_nth_page (this.page) as Document;
    }
    set
    {
      this.set_current_page (this.page_num (value));
    }
  }

  private List<unowned Document> _documents;
  /**
   * All opened documents.
   */
  public unowned List<unowned Document> documents
  {
    get
    {
      this._documents = (List<unowned Document>)this.get_children ();
      return this._documents;
    }
  }

  private void title_change (Document sender)
  {
    Gtk.HBox title;
    Gtk.Image image;
    Gtk.Label label;
    Gtk.Button button;

    title = new Gtk.HBox (false, 4);
    title.set_tooltip_markup (sender.get_tooltips ());

    image = new Gtk.Image.from_pixbuf (sender.get_icon ());
    title.pack_start (image, false, true, 0);

    label = new Gtk.Label (sender.title);
    title.pack_start (label, false, false, 0);

    /* Close button */
  	image = new Gtk.Image.from_stock (Gtk.Stock.CLOSE, Gtk.IconSize.MENU);
	  button = Utils.create_small_button (image);
    button.set_data ("document", sender);
    button.clicked.connect ((s) => {
      Document document;

      document = s.get_data<Document> ("document");
      document.close ();
    });
    button.set_tooltip_text (_("Close document"));
    title.pack_start (button, false, false, 0);

    title.show_all ();
    this.set_tab_label (sender, title);
    this.tab_state_changed (sender);
  }

  private void switch_page_cb (Gtk.Notebook sender, Gtk.NotebookPage page,
                               uint page_num)
  {
    this.current.setup_ui (this.ui_manager);
    this.tab_changed (this.current);
  }

  private void launch_external (string? filename) throws Error
  {
    if (filename != null)
    {
      string msg;
      Gtk.Window toplevel;
      Gtk.MessageDialog dialog;

      msg = _("%s doesn't support this type of document, would you like open this with the default program?").printf (Config.APPNAME);

      toplevel = this.get_toplevel () as Gtk.Window;
      dialog = new Gtk.MessageDialog (toplevel, Gtk.DialogFlags.MODAL,
                                      Gtk.MessageType.QUESTION,
                                      Gtk.ButtonsType.YES_NO, msg);

      if (dialog.run () == Gtk.ResponseType.YES)
      {
         // MANUEL BACHMANN : added commands for Win32
        if (Config.OS == "win32")
             Process.spawn_command_line_async ("cmd /c start \"" + filename + "\"");
        else
             AppInfo.launch_default_for_uri ("file://" + filename, null);
      }
      dialog.destroy ();
    }
  }

  /**
   * Update the user interface
   */
  protected virtual void setup_ui ()
  {
    bool active = false;

    if (this.current != null)
    {
      active = true;
    }
    this.ui_manager.get_action ("document-save").sensitive = active;
    this.ui_manager.get_action ("document-save-as").sensitive = active;
    this.ui_manager.get_action ("document-rename").sensitive = active;
    this.ui_manager.get_action ("document-close").sensitive = active;
    this.ui_manager.get_action ("edit-undo").sensitive = active;
    this.ui_manager.get_action ("edit-redo").sensitive = active;
    this.ui_manager.get_action ("edit-copy").sensitive = active;
    this.ui_manager.get_action ("edit-cut").sensitive = active;
    this.ui_manager.get_action ("edit-paste").sensitive = active;
    this.ui_manager.get_action ("edit-lower").sensitive = active;
    this.ui_manager.get_action ("edit-upper").sensitive = active;
    this.ui_manager.get_action ("search-find").sensitive = active;
    this.ui_manager.get_action ("search-find-next").sensitive = active;
    this.ui_manager.get_action ("search-find-prev").sensitive = active;
    this.ui_manager.get_action ("search-replace").sensitive = active;
    this.ui_manager.get_action ("search-goto-line").sensitive = active;
    this.ui_manager.get_action ("document-save-all").sensitive = active;
    this.ui_manager.get_action ("document-close-all").sensitive = active;
    this.ui_manager.get_action ("document-split").sensitive = active;
    if (this.current != null)
    {
      this.current.setup_ui (this.ui_manager);
    }
  }

  private void drag_data_received_cb (Gtk.Widget sender,
                                      Gdk.DragContext drag_context,
                                      int x, int y,
                                      Gtk.SelectionData data,
                                      uint info, uint time)
  {
    SList<string> files;

    files = new SList<string> ();
    foreach (string uri in ((string)data.data).split ("\r\n"))
    {
      try
      {
        if (uri != "")
        {
          File file;
          string filename;

          file = File.new_for_uri (uri);
          filename = file.get_path ();
          this.create (filename);
        }
      }
      catch (Error e)
      {
        debug (e.message);
      }
    }
  }

  private int find_tab_num_at_pos (int abs_x, int abs_y)
  {
    Gtk.Widget page;
    int page_num = 0;
    Gtk.PositionType tab_pos;

    tab_pos = this.get_tab_pos ();

    if (this.first_tab == null)
    {
      return -1;
    }

    while ((page = this.get_nth_page (page_num)) != null)
    {
      Gtk.Widget tab;
      int max_x, max_y;
      int x_root, y_root;

      tab = this.get_tab_label (page);
      return_val_if_fail (tab != null, -1);

      if (!tab.is_mapped ())
      {
        page_num++;
        continue;
      }

      tab.window.get_origin (out x_root, out y_root);

      max_x = x_root + tab.allocation.x + tab.allocation.width;
      max_y = y_root + tab.allocation.y + tab.allocation.height;

      if ((tab_pos == Gtk.PositionType.TOP || tab_pos == Gtk.PositionType.BOTTOM)
          && abs_x <= max_x)
      {
        return page_num;
      }
      else if ((tab_pos == Gtk.PositionType.LEFT || tab_pos == Gtk.PositionType.RIGHT)
               &&  abs_y <= max_y)
      {
        return page_num;
      }

      page_num++;
    }
    return -1;
  }

  private bool on_button_press (Gtk.Widget sender, Gdk.EventButton event)
  {
    bool catched = false;

    if (event.button == 3)
    {
      int tab_clicked;

      tab_clicked = this.find_tab_num_at_pos ((int)event.x_root, (int)event.y_root);
      if (tab_clicked == -1)
      {
        // CHECK: do we really need it?

        /* consume event, so that we don't pop up the context menu when
         * the mouse if not over a tab label
         */
        catched = true;
      }
      else
      {
        Gtk.Menu menu;

        /* Switch to the page the mouse is over, but don't consume the event */
        this.set_current_page (tab_clicked);

        menu = this.ui_manager.get_widget ("/document-popup") as Gtk.Menu;
        menu.popup (null, null, null, event.button, event.time);
      }
    }
    return catched;
  }

  /**
   * Create a new Valide.DocumentManager
   */
  public DocumentManager (UIManager ui_manager)
  {
    Object (ui_manager: ui_manager);
  }

  construct
  {
    Gtk.TargetEntry[] targets;

    this.scrollable = true;
    this.setup_ui ();
    this.switch_page.connect_after (this.switch_page_cb);
    this.button_press_event.connect (this.on_button_press);

    /* Seting up the drag & drop */
    targets = new Gtk.TargetEntry[1];
    targets[0].target = "text/uri-list";
    targets[0].flags = 0;
    targets[0].info = 0;

    Gtk.drag_dest_set (this, Gtk.DestDefaults.ALL, targets,
                       Gdk.DragAction.COPY);
    this.drag_data_received.connect (this.drag_data_received_cb);

    /* do not try to create a default doc. anymore */
    /* (with modifs, popped a "new document" window each time !) */
    /*try
    {
      this.create ();
    }
    catch (Error e)
    {
      debug (e.message);
    }*/
  }

  /**
   * Create a new document.
   *
   * @param filename a name of a file or null for create a new document.
   *
   * @return The new document.
   */
  public Document? create (string? filename = null) throws Error
  {
    int pos;
    string mime;
    string file_name;
    Document document;

    file_name = Utils.get_absolute_path (filename);
    if (!this.is_open (file_name, out pos))
    {
      if (file_name != null && this.get_n_pages () == 1)
      {
        document = this.current;
        if (document.is_new && !document.buffer.get_modified ())
        {
          document.close ();
        }
      }

      mime = Utils.get_mime_type (file_name);
      if (mime.has_prefix ("text/"))
      {
        document = new Document ();

         // if the document already exists => open
        if (file_name != null)
        {
          try
          {
            document.open (file_name);
          }
          catch (Error e)
          {
            warning (e.message);
          }
        }
         // if it doesn't => ask name before creating blank
        else
        {
           // MANUEL BACHMANN : new dialog to ask for file name
          var dialog = new DocumentNewDialog (document);
          dialog.entry.text = "New file.vala";
          dialog.entry.select_region (0,8);
          dialog.run ();
           //
           // do nothing if clicked on "Cancel"...
          if (document.path == "")
           return null;
        }

        document.title_change.connect (this.title_change);
        this.append_document (document);
        document.show_all ();

        document.buffer.notify["can-undo"].connect (() => {
          this.setup_ui ();
        });
        document.buffer.notify["can-redo"].connect (() => {
          this.setup_ui ();
        });
        document.buffer.changed.connect (() => {
          this.setup_ui ();
        });
        document.cursor_moved.connect ((s, r, c) => {
          this.tab_cursor_moved (s, r, c);
        });
        document.closed.connect ((s) => {
          this.remove_document (s);
        });
        document.saved.connect ((s) => {
          this.tab_saved (s);
        });
        document.state_change.connect ((sender, state) => {
          this.tab_state_changed (sender);
        });

        this.current = document;
        this.current.grab_focus ();
      }
      else
      {
        this.launch_external (file_name);
      }
    }
    else
    {
      this.set_current_page (pos);
    }

    Utils.process_gtk_events ();
    return this.current;
  }

  /**
   * Open document with #Gtk.FileChosserDialog.
   */
  public void open () throws Error
  {
    SList<string> files;
    Gtk.FileChooserDialog dialog;

    files = new SList<string> ();
    dialog = new Gtk.FileChooserDialog (_("Open file"), null,
                                        Gtk.FileChooserAction.OPEN,
                                        Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL,
                                        Gtk.Stock.OPEN, Gtk.ResponseType.ACCEPT,
                                        null);

    dialog.set_select_multiple (true);

      /* get last used save folder from config file,
      if blank use the user's home dir */
    string save_dir = ConfigManager.get_instance ().get_string ("General", "default-directory");
    if (save_dir != "")
      dialog.set_current_folder (save_dir);
    else
      dialog.set_current_folder (Environment.get_home_dir ());

    if (dialog.run () == Gtk.ResponseType.ACCEPT)
    {
      files = dialog.get_filenames ();
    }
    dialog.destroy ();

    foreach (string file in files)
    {
      Document document;

      document = this.create (file);
      if (document != null)
      {
        this.current.is_save = true;
      }
    }
  }

  /**
   * Save the current document.
   */
  public void save ()
  {
    if (this.current != null)
    {
       // MANUEL BACHMANN : pass tab title as potentiel name in "save" dialog
      this.current.save (this.current.title.replace ("* ",""));
    }
    else
    {
      message (_("No document open!"));
    }
  }

  /**
   * Save as the current document.
   */
  public void save_as ()
  {
    if (this.current != null)
    {
       // MANUEL BACHMANN : pass tab title as potentiel name in "save" dialog
      this.current.save_as (this.current.title.replace ("* ",""));
    }
    else
    {
      message (_("No document open!"));
    }
  }

  /**
   * Rename the current document.
   */
  public void rename ()
  {
    if (this.current != null)
    {
      this.current.rename ();
    }
    else
    {
      message (_("No document open!"));
    }
  }

  /**
   * Close the current document.
   */
  public void close ()
  {
    if (this.current != null)
    {
      this.current.close ();
    }
    else
    {
      message (_("No document open!"));
    }
  }

  /**
   * Save all opening documents.
   *
   * @return true if all documents are really saved.
   */
  public bool save_all ()
  {
    bool saved = true;

    if (this.current != null)
    {
      foreach (Document document in this.documents)
      {
        document.save ();
        if (!document.is_save)
        {
          saved = false;
          break;
        }
      }
    }
    else
    {
      message (_("No document open!"));
    }
    return saved;
  }

  /**
   * Close all opening documents.
   *
   * @return true if all documents are really closed.
   */
  public bool close_all ()
  {
    if (this.current != null)
    {
      int unsave = 0;
      bool close = true;

      foreach (Document document in this.documents)
      {
        if (!document.is_save)
        {
          unsave++;
        }
      }

      if (unsave > 0)
      {
        DocumentCloseDialog dialog;
        Gtk.Window parent;
        int response;

        dialog = new DocumentCloseDialog (this.documents);
        parent = this.get_ancestor (typeof (Gtk.Window)) as Gtk.Window;
        dialog.set_transient_for (parent);
        dialog.set_modal (true);
        response = dialog.run ();
        dialog.destroy ();

        switch (response)
        {
          case Gtk.ResponseType.OK:
            foreach (Document document in dialog.selected)
            {
              document.save ();
            }
            close = true;
          break;
          case Gtk.ResponseType.CLOSE:
            close = true;
          break;
          case Gtk.ResponseType.CANCEL:
            close = false;
          break;
        }
      }

      if (close)
      {
          /* this DOES NOT WORK and CRASHES the program,
             maybe due to quirks in the iterator... */
        /*List<Document> documents;

        documents = this.documents.copy ();
        foreach (Document document in documents)
        {
          document.is_save = true;
          document.close ();
        }*/

          /* ... so instead, we simply count how many
          documents are opened ("this.get_n_pages")
          and run "this.current.close" as much times,
          as when one is closed, another becomes current */        
	for (int i=0; i <= this.get_n_pages (); i++)
        {
          this.current.close ();
        }

      }
    }
    else
    {
      message (_("No document open!"));
    }
    return (this.get_n_pages () == 0);
  }

  public void split ()
  {
    if (this.current != null)
    {
      this.current.split ();
    }
    else
    {
      message (_("No document open!"));
    }
  }

  public void unsplit ()
  {
    if (this.current != null)
    {
      this.current.unsplit ();
    }
    else
    {
      message (_("No document open!"));
    }
  }

  /**
   * @see Gtk.Notebook.prev_page
   */
  public new void prev_page ()
  {
    if (this.get_current_page () == 0)
    {
      this.set_current_page (this.get_n_pages () - 1);
    }
    else
    {
      base.prev_page ();
    }
  }

  /**
   * @see Gtk.Notebook.next_page
   */
  public new void next_page ()
  {
    if (this.get_current_page () == (this.get_n_pages () - 1))
    {
      this.set_current_page (0);
    }
    else
    {
      base.next_page ();
    }
  }

  /**
   * Append new page in notebook.
   *
   * @param child the document added.
   */
  public int append_document (Document child)
  {
    int page_num;

    page_num = this.append_page (child, null);
    this.set_tab_reorderable (child, true);
    this.title_change (child);
    this.setup_ui ();
    this.tab_added (child);
    return page_num;
  }

  /**
   * Remove page from notebook.
   *
   * @param child a document.
   */
  public void remove_document (Document child)
  {
    int page_num;

    page_num = this.page_num (child);
    if (page_num >= 0)
    {
      this.remove_page (page_num);
      this.setup_ui ();
      this.tab_removed (child);
    }
  }

  /**
   * Returns the document contained in page number page_num.
   *
   * @param page_num the index of a page in the notebook, or -1 to get the last page. 
   *
   * @return The document, or NULL if page_num is out of bounds.
   */
  public Document get_nth (int page_num)
  {
    return this.get_nth_page (page_num) as Document;
  }

  /**
   * Check if file is already open.
   *
   * @param filename the path of file.
   * @param pos return tab position.
   *
   * @return true if file is already open.
   */
  public bool is_open (string? filename, out int pos)
  {
    bool opened = false;

    if (filename != null)
    {
      pos = 0;
      foreach (Document d in this.documents)
      {
        if (d.path == filename)
        {
          opened = true;
          break;
        }
        pos++;
      }
    }
    return opened;
  }
}
