/* abstract-document.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public enum Valide.DocumentState
{
  NORMAL,
  EXTERNALLY_MODIFIED,
  BAD_ENCODING,
  INFO
}

/**
 * The base class for text document
 */
public class Valide.Document : Gtk.VBox
{
  /**
   * The text view
   */
  public SplitSourceView split_view;

  private uint64 mtime;
  private Searching searching;

  /**
   * The cursor_moved signal will be emitted when the user moved the caret
   */
  public signal void cursor_moved (int row, int col);

  /**
   * The signal savec is emited when the document is saved
   */
  public signal void saved ();

  /**
   * The signal savec is emited when the document is closed
   */
  public signal void closed ();

  /**
   * The signal savec is emited when the title of the document changed
   */
  public signal void title_change ();

  /**
   * The signal savec is emited when the state of the document changed
   */
  public signal void state_change (DocumentState state);

  /**
   * The SourceBuffer
   */
  public SourceBuffer buffer
  {
    get
    {
      return this.text_view.source_buffer;
    }
  }

  /**
   * The current focus text view
   */
  public SourceView text_view
  {
    get
    {
      return this.split_view.active_view;
    }
  }

  /**
   * For compatibility with Gedit.Document
   */
  public SourceView view
  {
    get
    {
      return this.text_view;
    }
  }

  /**
   * Is that the document is newly created
   */
  public bool is_new
  {
    get
    {
       // return (this.path == _("New file.vala"));
       // MANUEL BACHMANN : as we now allow other creation names than "New file",
       //                   we must test if the file really exists
      var file = File.new_for_path (this.path);
      if (file.query_exists ())
       return false;
      else
       return true;
    }
  }

  private bool _save;
  /**
   * The document is uptodate with the hard disk copy?
   */
  public bool is_save
  {
    get
    {
      return this._save;
    }
    set
    {
      this._save = value;
      if (value)
      {
        this.saved ();
      }
      this.title_change ();
    }
  }

  private DocumentState _state;
  /**
   * The state of the document
   */
  public new DocumentState state
  {
    get
    {
      return this._state;
    }
    set
    {
      this._state = value;
      this.title_change ();
      switch (this.state)
      {
        case DocumentState.NORMAL:
          this.editable = true;
        break;
        case DocumentState.EXTERNALLY_MODIFIED:
        case DocumentState.BAD_ENCODING:
          this.editable = false;
        break;
      }
      this.state_change (this.state);
    }
  }

  /**
   * The document is editable?
   */
  public bool editable
  {
    get
    {
      return this.text_view.get_editable ();
    }
    set
    {
      this.text_view.set_editable (value);
    }
  }

  private string _selected_text = "";
  /**
   * The current selected text
   */
  public string selected_text
  {
    get
    {
      Gtk.TextIter start;
      Gtk.TextIter end;

      if (this.buffer.get_selection_bounds (out start, out end))
      {
        _selected_text = this.buffer.get_text (start, end, false);
      }
      return _selected_text;
    }
  }

  /*
   * Note: When the path is modified, the signal title_change has emitted and
   *       the old path is store in the object data.
   */

  /**
   * The absolute path of the document
   */
  public new string path
  {
    get
    {
      if (this.buffer.path == null)
      {
        return _("New file.vala");
      }
      else
      {
        return this.buffer.path;
      }
    }

    set
    {
      string old_path;

      old_path = this.buffer.path;
      this.set_data ("old_path", old_path);
      this.buffer.path = value;
      this.title_change ();
      this.set_data ("old_path", null);
    }
  }

  private string _filename;
  /**
   * The base name of the file (read-only)
   */
  public string filename
  {
    get
    {
      _filename = Path.get_basename (this.path);
      return _filename;
    }
  }

  private string _title;
  /**
   * The title of the document (read-only)
   */
  public string title
  {
    get
    {
      _title = "";
      if (this.buffer.get_modified ())
      {
        _title = "* ";
      }
      _title += this.filename;
      return _title;
    }
  }

  private void check_externally_modified ()
  {
    if (!this.is_new && this.state == DocumentState.NORMAL)
    {
      try
      {
        uint64 mtime;

        mtime = Utils.get_mtime (this.path);
        if (this.mtime < mtime)
        {
          DocumentMessage msg;

          this.state = DocumentState.EXTERNALLY_MODIFIED;
          msg = new DocumentMessage (this);
          this.pack_start (msg, false, true, 0);
          this.reorder_child (msg, 0);
          msg.show_all ();
        }
      }
      catch (Error e)
      {
        debug (e.message);
      }
    }
  }

  private void scroll_to_iter (Gtk.TextIter iter)
  {
    Gtk.TextMark mark;

    mark = this.buffer.create_mark ("cursor", iter, true);
    this.text_view.scroll_to_mark (mark, 0.4, false, 0, 0);
    this.buffer.delete_mark (mark);
  }

  private void cursor_move ()
  {
    int row;
    int col;
    Gtk.TextIter iter;
    Gtk.TextIter start;
    uint tab_size;

    buffer.get_iter_at_mark (out iter, this.buffer.get_insert ());

    row = iter.get_line ();

    start = iter;
    start.set_line_offset (0);
    col = 0;

    tab_size = this.text_view.get_tab_width ();

    while (!start.equal (iter))
    {
      /* FIXME: Are we Unicode compliant here? */
      if (start.get_char () == '\t')
      {
        col += (int)(tab_size - (col  % tab_size));
      }
      else
      {
        col++;
      }

      start.forward_char ();
    }

    this.cursor_moved (row + 1, col + 1);
  }

  private void on_action_search ()
  {
    this.searching.find ();
  }

  private void on_action_find_next ()
  {
    this.searching.find_next ();
  }

  private void on_action_find_prev ()
  {
    this.searching.find_prev ();
  }

  private void on_action_replace ()
  {
    this.searching.replace ();
  }

  /**
   * Return the entire file content in UTF-8 encoding
   *
   * @return The content of the document
   */
  protected virtual string get_contents () throws Error
  {
    return this.buffer.get_contents ();
  }

  /**
   * Write the entire document content in file
   */
  protected virtual void set_contents (string? new_path = null) throws Error
  {
    this.buffer.set_contents (new_path);
  }

  construct
  {
    this.is_save = true;
    this.split_view = new SplitSourceView ();
    this.path = null;
    this.pack_start (this.split_view, true, true, 0);
    this.buffer.changed.connect (() => {
      this.cursor_move ();
    });
    this.buffer.modified_changed.connect (() => {
      this.is_save = !this.buffer.get_modified ();
      this.title_change ();
    });
    this.buffer.mark_set.connect (() => {
      this.cursor_move ();
    });

    this.searching = new Searching (this);
    this.pack_start (this.searching, false, true, 0);
    this.searching.hide ();

    this.cursor_move ();

    this.state = DocumentState.NORMAL;
    this.text_view.focus_in_event.connect (() => {
      this.check_externally_modified ();
      this.cursor_move ();
      return false;
    });

    this.split_view.search.connect (this.on_action_search);
    this.split_view.find_next.connect (this.on_action_find_next);
    this.split_view.find_prev.connect (this.on_action_find_prev);
    this.split_view.replace.connect (this.on_action_replace);
  }

  /**
   * @see Gtk.Widget.grab_focus
   */
  public override void grab_focus ()
  {
    this.text_view.grab_focus ();
  }

  /**
   * @see Gtk.Widget.show
   */
  public override void show ()
  {
    base.show ();
    this.searching.hide ();
  }

  /**
   * @see Gtk.Widget.show_all
   */
  public override void show_all ()
  {
    base.show_all ();
    this.searching.hide ();
  }

  /**
   * Refresh the content of the widget with the hard disk file
   */
  public void reload ()
  {
    string contents = "";

    try
    {
      contents = this.get_contents ();
    }
    catch (Error e)
    {
      DocumentMessage msg;

      this.state = DocumentState.BAD_ENCODING;
      msg = new DocumentMessage (this);
      this.pack_start (msg, false, true, 0);
      this.reorder_child (msg, 0);
      msg.show_all ();
    }

    Gtk.TextIter end;
    Gtk.TextIter start;

    this.buffer.get_start_iter (out start);
    this.buffer.get_end_iter (out end);
    this.buffer.begin_not_undoable_action ();
    this.buffer.delete (start, end);
    this.buffer.insert (start, contents, -1);
    this.buffer.set_modified (false);
    this.buffer.end_not_undoable_action ();
    this.is_save = true;
    try
    {
      this.mtime = Utils.get_mtime (this.path);
    }
    catch (Error e)
    {
      debug (e.message);
    }
  }

  /**
   * Open a document
   *
   * @param path A path of a file
   */
  public void open (string path) throws DocumentError
  {
    if (FileUtils.test (path, FileTest.EXISTS))
    {
      this.path = path;
      this.reload ();
    }
    else
    {
      throw new DocumentError.BAD_URI (_("cannot open file '%s': No such file or directory"), path);
    }
  }

  /**
   * Save the content of the widget in hard file file.
   * If the file has not yet been saved, this function show a dialog for enter
   * the path of the file.
   *
   * @param current_folder The current folder of chooser
   */
   // MANUEL BACHMANN : additionnal parameter "current_name"
  public void save (string? current_name = null, string? current_folder = null)
  {
    int response = Gtk.ResponseType.ACCEPT;

    if (this.is_new)
    {
      Gtk.FileChooserDialog dialog;

      dialog = new Gtk.FileChooserDialog (_("Save file"), null,
                                          Gtk.FileChooserAction.SAVE,
                                          Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL,
                                          Gtk.Stock.SAVE, Gtk.ResponseType.ACCEPT,
                                          null);

      if (current_name != null)
      {
        dialog.set_current_name (current_name);
      }
      if (current_folder != null)
      {
        dialog.set_current_folder (current_folder);
      }
      else
      {
          /* get last used save folder from config file,
          if blank use the user's home dir */
        string save_dir = ConfigManager.get_instance ().get_string ("General", "default-directory");
        if (save_dir != "")
          dialog.set_current_folder (save_dir);
        else
          dialog.set_current_folder (Environment.get_home_dir ());
      }
      response = dialog.run ();
      if (response == Gtk.ResponseType.ACCEPT)
      {
        this.path = dialog.get_filename ();
         // save the save folder to config file
        ConfigManager.get_instance ().set_string ("General", "default-directory", dialog.get_current_folder ());
      }
      dialog.destroy ();
    }
    if (response == Gtk.ResponseType.ACCEPT)
    {
      try
      {
        this.set_contents (this.path);
        this.buffer.set_modified (false);
        this.mtime = Utils.get_mtime (this.path);
        this.is_save = true;
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
  }

  /**
   * Save as the content of the widget in hard disk file
   *
   * @param current_folder The current folder of chooser
   */
   // MANUEL BACHMANN : additionnal parameter "current_name"
  public void save_as (string? current_name = null, string? current_folder = null)
  {
    string path = null;
    int response = Gtk.ResponseType.ACCEPT;

    Gtk.FileChooserDialog dialog;

    dialog = new Gtk.FileChooserDialog (_("Save as file"), null,
                                        Gtk.FileChooserAction.SAVE,
                                        Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL,
                                        Gtk.Stock.SAVE, Gtk.ResponseType.ACCEPT,
                                        null);

     // ask for confirmation
    dialog.set_do_overwrite_confirmation (true);

    if (current_name != null)
    {
      dialog.set_current_name (current_name);
    }
    if (current_folder != null)
    {
      dialog.set_current_folder (current_folder);
    }
    else
    {
        /* get last used save folder from config file,
        if blank use the user's home dir */
      string save_dir = ConfigManager.get_instance ().get_string ("General", "default-directory");
      if (save_dir != "")
        dialog.set_current_folder (save_dir);
      else
        dialog.set_current_folder (Environment.get_home_dir ());
    }
    response = dialog.run ();
    if (response == Gtk.ResponseType.ACCEPT)
    {
      path = dialog.get_filename ();
       // save the save folder to config file
      ConfigManager.get_instance ().set_string ("General", "default-directory", dialog.get_current_folder ());
    }
    dialog.destroy ();

    if (response == Gtk.ResponseType.ACCEPT)
    {
      try
      {
        this.set_contents (path);
        this.path = path;
        this.buffer.set_modified (false);
        this.mtime = Utils.get_mtime (this.path);
        this.is_save = true;
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
  }

  /**
   * Rename the hard disk file
   */
  public void rename ()
  {
    Gtk.Entry entry;
    Gtk.Dialog dialog;
    Gtk.Window parent;

    dialog = new Gtk.Dialog ();
    dialog.set_title (_("Rename"));
    parent = this.get_ancestor (typeof (Gtk.Window)) as Gtk.Window;
    dialog.set_transient_for (parent);
    dialog.set_modal (true);
    dialog.add_button (Gtk.Stock.APPLY, Gtk.ResponseType.APPLY);
    dialog.add_button (Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL);

    entry = new Gtk.Entry ();
    entry.set_data ("dialog", dialog);
    entry.activate.connect ((s) => {
      Gtk.Dialog dlg;

      dlg = s.get_data<Gtk.Dialog> ("dialog");
      dlg.response (Gtk.ResponseType.APPLY);
    });
    entry.set_text (this.path);
    dialog.vbox.pack_start (entry, true, true, 10);

    dialog.show_all ();
    if (dialog.run () == Gtk.ResponseType.APPLY)
    {
      string new_path;

      new_path = entry.get_text ();
      FileUtils.rename (this.path, new_path);
      this.path = new_path;
    }
    dialog.destroy ();
  }

  /**
   * Close the document
   */
  public bool close ()
  {
    bool is_close = true;

    if (!this.is_save)
    {
      Gtk.Window parent;
      Gtk.Dialog dialog;
      Gtk.Label label;
      int response;

      dialog = new Gtk.Dialog ();
      dialog.set_title (_("Are you sure?"));
      dialog.add_button (Gtk.Stock.YES, Gtk.ResponseType.YES);
      dialog.add_button (Gtk.Stock.NO, Gtk.ResponseType.NO);
      dialog.add_button (Gtk.Stock.CANCEL, Gtk.ResponseType.CANCEL);
      dialog.set_modal (true);
      parent = this.get_ancestor (typeof (Gtk.Window)) as Gtk.Window;
      dialog.set_transient_for (parent);

      label = new Gtk.Label (_("The document has unsaved changes. Save changes before closing?"));
      dialog.vbox.pack_start (label, true, false, 10);
      dialog.show_all ();
      response = dialog.run ();
      dialog.destroy ();
      switch (response)
      {
        case Gtk.ResponseType.YES:
          this.save ();
          is_close = this.is_save;
        break;
        case Gtk.ResponseType.NO:
        break;
        case Gtk.ResponseType.CANCEL:
          is_close =  false;
        break;
      }
    }
    if (is_close)
    {
      this.closed ();
    }
    return is_close;
  }

  /**
   * Select the specified offsets
   *
   * @param start Char offset from start of the buffer
   * @param end Char offset from start of the buffer
   *
   */
  public void select_offsets (int start, int end)
  {
    Gtk.TextIter end_iter;
    Gtk.TextIter start_iter;

    this.buffer.get_iter_at_offset (out start_iter, start);
    this.buffer.get_iter_at_offset (out end_iter, end);
    this.buffer.select_range (start_iter, end_iter);
    this.scroll_to_iter (start_iter);
  }

  /**
   * Get the pixbuf of the file
   *
   * @return A Gdk.Pixbuf
   */
  public virtual Gdk.Pixbuf get_icon ()
  {
    Gdk.Pixbuf pixbuf;

    switch (this.state)
    {
      case DocumentState.NORMAL:
        try
        {
          pixbuf = Utils.get_pixbuf_for_file (this.path, Gtk.IconSize.MENU);
        }
        catch (Error e)
        {
          pixbuf = Utils.get_pixbuf_for_stock (Gtk.Stock.FILE, Gtk.IconSize.MENU);
        }
      break;
      case DocumentState.EXTERNALLY_MODIFIED:
        pixbuf = Utils.get_pixbuf_for_stock (Gtk.Stock.DIALOG_WARNING, Gtk.IconSize.MENU);
      break;
      case DocumentState.BAD_ENCODING:
        pixbuf = Utils.get_pixbuf_for_stock (Gtk.Stock.DIALOG_ERROR, Gtk.IconSize.MENU);
      break;
      default:
        pixbuf = Utils.get_pixbuf_for_stock (Gtk.Stock.DIALOG_QUESTION, Gtk.IconSize.MENU);
      break;
    }
    return pixbuf;
  }

  /**
   * Get the informations about this file
   *
   * @return An information text
   */
  public virtual string get_tooltips ()
  {
    string tooltips = null;

    if (!this.is_new)
    {
      string mime_type;
      string content_type;
      string content_description = null;

      try
      {
        content_type = Utils.get_mime_type (this.path);
        content_description = ContentType.get_description (content_type);
      }
      catch (Error e)
      {
        debug (e.message);
        content_type = _("Unknow");
      }
      if (content_description != null)
      {
        mime_type = "%s (%s)".printf (content_description, content_type);
      }
      else
      {
        mime_type = content_type;
      }

      tooltips = "<b>%s</b> %s\n\n<b>%s</b> %s\n<b>%s</b> %s".printf (_("Name:"), this.path,
                                                                      _("MIME Type:"), mime_type,
                                                                      _("Encoding:"), this.buffer.encoding);
    }
    return tooltips;
  }

  /**
   * Update the user interface
   *
   * @param ui_manager The UI manager
   */
  public void setup_ui (UIManager ui_manager)
  {
    ui_manager.get_action ("edit-undo").sensitive = this.buffer.can_undo;
    ui_manager.get_action ("edit-redo").sensitive = this.buffer.can_redo;
    ui_manager.action_set_toggled ("document-split", this.split_view.is_split);
  }

  /**
   * Split the document container
   */
  public void split ()
  {
    this.split_view.split ();
  }

  /**
   * Unsplit the document container
   */
  public void unsplit ()
  {
    this.split_view.unsplit ();
  }

  /**
   * Select a range
   *
   * @param pos The position of the beginning and end of the selection
   */
  public void select_range (Position pos)
  {
    Gtk.TextIter start;
    Gtk.TextIter end;

    if (pos.start_line >= 0)
    {
      this.buffer.get_start_iter (out start);
      start.set_line (pos.start_line);
      if (pos.start_row >= 0 && pos.start_row < start.get_chars_in_line ())
      {
        start.set_line_offset (pos.start_row);
      }
      end = start;
      if (pos.end_line >= 0 && pos.end_line < this.buffer.get_line_count ())
      {
        end.set_line (pos.end_line);
        if (pos.end_row >= 0 && pos.end_row < end.get_chars_in_line ())
        {
          end.set_line_offset (pos.end_row);
        }
      }

      this.buffer.select_range (start, end);
      this.scroll_to_iter (start);
    }
    else
    {
      debug (_("Invalid range"));
    }
  }
}

