/* window.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 * 	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public class Valide.FullscreenToolbar : Gtk.Window
{
  private bool animation_enter;
  private uint animation_timeout_id;

  public new Window parent
  {
    get;
    construct;
  }

  private bool on_enter_notify_event (Gtk.Widget sender, Gdk.EventCrossing event)
  {
    this.show_hide (true, 0);
    return false;
  }

  private bool on_leave_notify_event (Gtk.Widget sender, Gdk.EventCrossing event)
  {
    int w, h;
    int x, y;
    Gdk.Screen screen;
    Gdk.Display display;

    display = Gdk.Display.get_default ();
    screen = this.parent.get_screen ();

    this.get_size (out w, out h);
    display.get_pointer (out screen, out x, out y, null);

    /*
     * gtk seems to emit leave notify when clicking on tool items,
     * work around it by checking the coordinates
     */
    if (y >= h)
    {
      this.show_hide (false, h);
    }

    return false;
  }

  private bool run_animation ()
  {
    int x, y;
    Gdk.Screen screen;
    Gdk.Rectangle fs_rect;

    screen = this.parent.get_screen ();
    screen.get_monitor_geometry (screen.get_monitor_at_window (this.parent.window),
                                 out fs_rect);

    this.get_position (out x, out y);

    if (this.animation_enter)
    {
      if (y == fs_rect.y)
      {
        this.animation_timeout_id = 0;
        return false;
      }
      else
      {
        this.move (x, y + 1);
        return true;
      }
    }
    else
    {
      int w, h;

      this.get_size (out w, out h);

      if (y == fs_rect.y - h + 1)
      {
        this.animation_timeout_id = 0;
        return false;
      }
      else
      {
        this.move (x, y - 1);
        return true;
      }
    }
  }

  private void show_hide (bool show, int height)
  {
    Gtk.Settings settings;
    bool enable_animations;

    settings = this.parent.get_settings ();
    enable_animations = settings.gtk_enable_animations;

    if (enable_animations)
    {
      this.animation_enter = show;

      if (this.animation_timeout_id == 0)
      {
        this.animation_timeout_id = Timeout.add (4, this.run_animation);
      }
    }
    else
    {
      Gdk.Screen screen;
      Gdk.Rectangle fs_rect;

      screen = this.parent.get_screen ();
      screen.get_monitor_geometry (screen.get_monitor_at_window (this.parent.window),
                                   out fs_rect);

      if (show)
      {
        this.move (fs_rect.x, fs_rect.y);
      }
      else
      {
        this.move (fs_rect.x, fs_rect.y - height + 1);
      }
    }

  }

  public FullscreenToolbar (Window parent)
  {
    Object (parent: parent);
  }

  construct
  {
    Gtk.Window window;
    Gtk.Action action;
    Gtk.Toolbar toolbar;

    this.type = Gtk.WindowType.POPUP;
    this.set_transient_for (this.parent);

    toolbar = this.parent.ui_manager.get_widget ("/fullscreen-toolbar") as Gtk.Toolbar;
    toolbar.set_style (Gtk.ToolbarStyle.BOTH_HORIZ);
    this.add (toolbar);

    action = this.parent.ui_manager.get_action ("window-leave-fullscreen");
    action.set ("is_important", true);

    this.enter_notify_event.connect (this.on_enter_notify_event);
    this.leave_notify_event.connect (this.on_leave_notify_event);
  }

  public override void show ()
  {
    int w, h;
    Gdk.Screen screen;
    Gdk.Rectangle fs_rect;

    screen = this.parent.get_screen ();
    screen.get_monitor_geometry (screen.get_monitor_at_window (this.parent.window),
                                 out fs_rect);

    this.get_size (out w, out h);
    this.resize (fs_rect.width, h);
    this.move (fs_rect.x, fs_rect.y - h + 1);
    base.show ();
  }
}

