/* native-builder.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

private class Valide.HelpDialog : Gtk.Dialog
{
  public HelpDialog ()
  {
    string message;
    Gtk.Label label;

    this.add_button (Gtk.Stock.CLOSE, Gtk.ResponseType.CLOSE);
    this.set_modal (true);
    this.set_border_width (5);

    label = new Gtk.Label ("");
    this.vbox.add (label);
    label.show ();

    try
    {
      Process.spawn_command_line_sync ("valac --help", out message, null, null);
    }
    catch (Error e)
    {
      debug (e.message);
      message = e.message;
    }
    label.set_text (message);
  }

  public new int run ()
  {
    this.show_all ();
    return base.run ();
  }
}

public class Valide.NativeBuilderPreferences : AbstractNativeBuilderPreferences
{
  private Gtk.ListStore list_store_src;
  private Gtk.ListStore list_store_dst;
  private Gtk.ListStore list_store_vapi;
  private unowned NativeBuilderOptions options;

  public Project project
  {
    get;
    construct;
  }

  public Gtk.Widget widget
  {
    get
    {
      return this.widgets.notebook;
    }
  }

  private List<string> _pkg;
  /**
   * The list of the packages
   */
  public List<string> pkg
  {
    get
    {
      Gtk.TreeIter iter;
      string str;

      _pkg = new List<string> ();
      if (this.list_store_dst.get_iter_first (out iter))
      {
        do
        {
          this.list_store_dst.get (iter, 0, out str);
          _pkg.append (str);
        } while (this.list_store_dst.iter_next (ref iter));
      }
      return _pkg;
    }
  }

  private List<string> _vapi_dir;
  /**
   * The list of the vapi directory
   */
  public List<string> vapi_dir
  {
    get
    {
      Gtk.TreeIter iter;
      string str;

      _vapi_dir = new List<string> ();
      if (this.list_store_vapi.get_iter_first (out iter))
      {
        do
        {
          this.list_store_vapi.get (iter, 0, out str);
          _vapi_dir.append (str);
        } while (this.list_store_vapi.iter_next (ref iter));
      }
      return _vapi_dir;
    }
  }

  private bool pkg_exist (string pkg)
  {
    bool exist = false;

    foreach (string p in this.pkg)
    {
      if (p == pkg)
      {
        exist = true;
        break;
      }
    }
    return exist;
  }

  private void populate_tree_view (Gtk.TreeView tree_view)
  {
    Gtk.ListStore list_store;
    Gtk.CellRendererText render;
    Gtk.TreeViewColumn column;

    list_store = new Gtk.ListStore (1, typeof (string));
    list_store.set_sort_column_id (0, Gtk.SortType.ASCENDING);

    tree_view.set_model (list_store);
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes ("", render, "text", 0, null);
    tree_view.append_column (column);
    tree_view.headers_visible = false;
  }

  private void update_vapi ()
  {
    List<string> pkg;

    pkg = Valac.get_package_list (this.vapi_dir);
    this.list_store_src.clear ();
    foreach (string p in pkg)
    {
      Gtk.TreeIter iter;

      this.list_store_src.append (out iter);
      this.list_store_src.set (iter, 0, p);
    }
  }

  /* @todo Check duplicated items */
  /**
   * Callback for add a vapi directory
   */
  [CCode (instance_pos = -1)]
  protected void add_vapi_dir (Gtk.Widget sender)
  {
    string dirname;

    dirname = this.widgets.file_chooser_button.get_filename ();
    if (dirname != "")
    {
      Gtk.TreeIter iter;

      this.list_store_vapi.append (out iter);
      this.list_store_vapi.set (iter, 0, dirname);
      this.project.vapi_dir.append (new VapiDir (dirname));
      this.project.save ();
      this.update_vapi ();
    }
  }

  /**
   * Callback for delete a vapi directory
   */
  [CCode (instance_pos = -1)]
  protected void del_vapi_dir (Gtk.Widget sender)
  {
    Gtk.TreeIter iter;
    Gtk.TreeSelection selection;

    selection = this.widgets.tree_view_vapi.get_selection ();
    if (selection.get_selected (null, out iter))
    {
      string str;

      this.list_store_vapi.remove (iter);
      this.project.vapi_dir = new List<VapiDir> ();
      if (this.list_store_vapi.get_iter_first (out iter))
      {
        do
        {
          this.list_store_vapi.get (iter, 0, out str);
          this.project.vapi_dir.append (new VapiDir (str));
        } while (this.list_store_vapi.iter_next (ref iter));
      }
      this.project.save ();
      this.update_vapi ();
    }
  }

  /**
   * Callback for add a package with double-clic
   */
  [CCode (instance_pos = -1)]
  protected void row_add_pkg (Gtk.TreeView sender,
                              Gtk.TreePath path,
                              Gtk.TreeViewColumn column)
  {
    this.add_pkg (sender);
  }

  /**
   * Callback for delete a package with double-clic
   */
  [CCode (instance_pos = -1)]
  protected void row_del_pkg (Gtk.TreeView sender,
                              Gtk.TreePath path,
                              Gtk.TreeViewColumn column)
  {
    this.del_pkg (sender);
  }

  /**
   * Callback for add a package
   */
  [CCode (instance_pos = -1)]
  protected void add_pkg (Gtk.Widget sender)
  {
    string pkg;
    Gtk.TreeIter iter;
    Gtk.TreeSelection selection;

    selection = this.widgets.tree_view_src.get_selection ();
    if (selection.get_selected (null, out iter))
    {
      this.list_store_src.get (iter, 0, out pkg);
      if (!this.pkg_exist (pkg))
      {
        this.list_store_dst.append (out iter);
        this.list_store_dst.set (iter, 0, pkg);
        this.project.packages.append (new Package (pkg));
        this.project.save ();
      }
    }
  }

  /**
   * Callback for delete a package
   */
  [CCode (instance_pos = -1)]
  protected void del_pkg (Gtk.Widget sender)
  {
    Gtk.TreeIter iter;
    Gtk.TreeSelection selection;

    selection = this.widgets.tree_view_dst.get_selection ();
    if (selection.get_selected (null, out iter))
    {
      string str;

      this.list_store_dst.remove (iter);
      this.project.packages = new List<Package> ();
      if (this.list_store_dst.get_iter_first (out iter))
      {
        do
        {
          this.list_store_dst.get (iter, 0, out str);
          this.project.packages.append (new Package (str));
        } while (this.list_store_dst.iter_next (ref iter));
      }
      this.project.save ();
    }
  }

  /**
   * Callback for show the valac help
   */
  [CCode (instance_pos = -1)]
  protected void show_help (Gtk.Button sender)
  {
    HelpDialog help_dialog;

    help_dialog = new HelpDialog ();
    help_dialog.run ();
    help_dialog.destroy ();
  }

  public NativeBuilderPreferences (Project project)
  {
    Object (project: project);
  }

  construct
  {
    Gtk.TreeIter iter;

    this.options = this.project.builder_options as NativeBuilderOptions;

    this.populate_tree_view (this.widgets.tree_view_vapi);
    this.list_store_vapi = this.widgets.tree_view_vapi.model as Gtk.ListStore;
    this.populate_tree_view (this.widgets.tree_view_src);
    this.list_store_src = this.widgets.tree_view_src.model as Gtk.ListStore;
    this.populate_tree_view (this.widgets.tree_view_dst);
    this.list_store_dst = this.widgets.tree_view_dst.model as Gtk.ListStore;

    this.widgets.debug.active = this.options.debug;
    this.widgets.debug.toggled.connect ((s) => {
      this.options.debug = s.active;
      this.project.save ();
    });

    this.widgets.thread.active = this.options.thread;
    this.widgets.thread.toggled.connect ((s) => {
      this.options.thread = s.active;
      this.project.save ();
    });

    this.widgets.disable_assert.active = this.options.disable_assert;
    this.widgets.disable_assert.toggled.connect ((s) => {
      this.options.disable_assert = s.active;
      this.project.save ();
    });

    this.widgets.disable_checking.active = this.options.disable_checking;
    this.widgets.disable_checking.toggled.connect ((s) => {
      this.options.disable_checking = s.active;
      this.project.save ();
    });

    this.widgets.disable_non_null.active = this.options.disable_non_null;
    this.widgets.disable_non_null.toggled.connect ((s) => {
      this.options.disable_non_null = s.active;
      this.project.save ();
    });

    this.widgets.save_temps.active = this.options.save_temps  ;
    this.widgets.save_temps.toggled.connect ((s) => {
      this.options.save_temps = s.active;
      this.project.save ();
    });

    this.widgets.quiet.active = this.options.quiet;
    this.widgets.quiet.toggled.connect ((s) => {
      this.options.quiet = s.active;
      this.project.save ();
    });

    this.widgets.other.set_text (this.options.other);
    this.widgets.other.changed.connect ((s) => {
      this.options.other = (s as Gtk.Entry).get_text ();
      this.project.save ();
    });

    foreach (Package package in this.project.packages)
    {
      this.list_store_dst.append (out iter);
      this.list_store_dst.set (iter, 0, package.name);
    }

    foreach (VapiDir vapi_dir in this.project.vapi_dir)
    {
      this.list_store_vapi.append (out iter);
      this.list_store_vapi.set (iter, 0, vapi_dir.path);
    }

    this.update_vapi ();
    this.connect_signals ("valide_native_builder_preferences_");
  }
}

public class Valide.NativeBuilderOptions : Object,
                                           BuilderOptions, GLib.YAML.Buildable
{
  /**
   * Produce debug information
   */
  public bool debug { get; set; }

  /**
   * Enable multithreading support
   */
  public bool thread { get; set; }

  /**
   * Disable assertions
   */
  public bool disable_assert { get; set; }

  /**
   * Disable additional run-time checks
   */
  public bool disable_checking { get; set; }

  /**
   * disable experimental enhancements for non-null types
   */
  public bool disable_non_null { get; set; }

  /**
   * Keep temporary files
   */
  public bool save_temps { get; set; }

  /**
   * Do not print messages to the console
   */
  public bool quiet { get; set; }

  /**
   * Other options
   */
  public string other { get; set; }

  construct
  {
    this.other = "";
  }
}

/**
 * The native builder
 */
public class Valide.NativeBuilder : Builder, Object
{
  /**
   * @see Valide.Builder.project
   */
  public Project project { get; construct set; }

  /**
   * @see Valide.Builder.executables
   */
  public ExecutableManager executables { get; construct set; }

  private static NativeBuilderPreferences preferences_widget;

  private string get_compiler_options ()
  {
    string opt = "";
    NativeBuilderOptions options;

    opt += "-o \"%s\"".printf (this.project.get_executable_name ());

    foreach (Package package in this.project.packages)
    {
      opt += " --pkg=" + package.name;
    }

    foreach (VapiDir vapi_dir in this.project.vapi_dir)
    {
      opt += " --vapidir=\"" + vapi_dir.path + "\"";
    }

    options = this.project.builder_options as NativeBuilderOptions;
    if (options.debug)
    {
      opt += " -g";
    }

    if (options.thread)
    {
      opt += " --thread";
    }

    if (options.disable_assert)
    {
      opt += " --disable-assert";
    }

    if (options.disable_checking)
    {
      opt += " --disable-checking";
    }

    if (options.disable_non_null)
    {
      opt += " --disable-non-null";
    }

    if (options.save_temps)
    {
      opt += " --save-temps";
    }

    if (options.quiet)
    {
      opt += " -q";
    }

    opt += " " + options.other;

    foreach (Source file in this.project.files)
    {
      if (file.path.has_suffix (".vala")
          || file.path.has_suffix (".gs")
          || file.path.has_suffix (".c"))
      {
        opt += " \"%s\"".printf (file.path);
      }
    }
    return opt;
  }

  public NativeBuilder (Project project)
  {
    Object (project: project);
  }

  /**
   * @see Valide.Builder.configure
   */
  public int configure ()
  {
    message (_("This project couldn't be configurate, choose another builder"));
    return -1;
  }

  /**
   * @see Valide.Builder.build
   */
  public int build ()
  {
    Compiler compiler;
    ExecutableOptions options;

    options = new ExecutableOptions ();
    options.arguments = this.get_compiler_options ();
    options.working_dir = this.project.path;
    compiler = new Compiler (this.project);
    return executables.run (compiler, options);
  }

  /**
   * @see Valide.Builder.install
   */
  public int install ()
  {
    message (_("This project couldn't be install, choose another builder"));
    return -1;
  }

  /**
   * @see Valide.Builder.dist
   */
  public int dist ()
  {
    message (_("This project couldn't be distribute, choose another builder"));
    return -1;
  }

  /**
   * @see Valide.Builder.clean
   */
  public int clean ()
  {
    FileUtils.remove (project.get_executable_name ());
    return -1;
  }

  /**
   * @see Valide.Builder.distclean
   */
  public int distclean ()
  {
    string filename;

    foreach (Source file in this.project.files)
    {
      if (file.path.has_suffix (".vala")
          || file.path.has_suffix (".gs"))
      {
        filename = this.project.get_real_filename (file.path);
        filename = Utils.skip_extension (filename);
        FileUtils.remove (filename + ".c");
        FileUtils.remove (filename + ".h");
      }
    }
    return this.clean ();
  }

  /**
   * @see Valide.Builder.uninstall
   */
  public int uninstall ()
  {
    message (_("This project couldn't be uninstall, choose another builder"));
    return -1;
  }

  /**
   * @see Valide.Builder.preferences
   */
  public Gtk.Widget preferences ()
  {
    this.preferences_widget = new NativeBuilderPreferences (this.project);
    return this.preferences_widget.widget;
  }
}

