/* plugin-preferences.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/**
 * The widget for plugin preferences
 */
public class Valide.PluginPreferences : Gtk.VBox
{
  /* enum for tree model columns, not all is currently used */
  private enum Col
  {
	  ACTIVE,
	  INFO,
    NB
  }

  private Gtk.TreeView tree_view;
  private Gtk.Button about_button;
  private Gtk.Button configure_button;

  /**
   * The plugin manager
   */
  public PluginManager plugin_manager
  {
    get;
    construct;
  }

  private PluginRegistrar get_selected_plugin()
  {
    Gtk.TreeIter iter;
    Gtk.TreeModel model;
    PluginRegistrar pr = null;
    Gtk.TreeSelection tree_selection;

    tree_selection = this.tree_view.get_selection ();
    if (tree_selection.get_selected (out model, out iter))
    {
      model.get (iter, Col.INFO, out pr);
    }
    return pr;
  }

  private void active_toggled (Gtk.CellRendererToggle sender, string path_str)
  {
    Gtk.TreeIter iter;
    Gtk.TreePath path;
    Gtk.TreeModel model;

    path = new Gtk.TreePath.from_string (path_str);

    model = this.tree_view.get_model ();
    return_if_fail (model != null);

    model.get_iter (out iter, path);

    this.toggle_active (iter, model);
  }

  private void info_cell (Gtk.CellLayout cell_layout, Gtk.CellRenderer cell,
                          Gtk.TreeModel tree_model, Gtk.TreeIter iter)
  {
  	PluginRegistrar pr;

  	tree_model.get (iter, Col.INFO, out pr);
    if (pr != null)
	  {
      (cell as Gtk.CellRendererText).markup = GLib.Markup.printf_escaped ("<b>%s</b>\n%s",
                                                                          pr.name,
                                                                          pr.description);
      cell.sensitive = pr.is_available;
    }
  }

  private int model_name_sort (Gtk.TreeModel model,
                               Gtk.TreeIter a, Gtk.TreeIter b)
  {
    return 0;
  }

  private static bool name_search (Gtk.TreeModel model, int column, string key,
                                   Gtk.TreeIter iter)
  {
	  PluginRegistrar pr;
	  bool retval = false;

	  model.get (iter, Col.INFO, out pr);
	  if (pr != null)
	  {
	    retval = !(pr.name.casefold () == key.casefold ());
	  }
	  return retval;
  }

  private void cursor_changed (Gtk.TreeView sender)
  {
  	PluginRegistrar pr;

  	pr = this.get_selected_plugin ();
    if (pr != null)
    {
  	  this.about_button.sensitive = pr.is_active;
  	  this.configure_button.sensitive = (pr.is_active && pr.is_configurable);
  	}
  }

  private void row_activated (Gtk.TreeView sender, Gtk.TreePath path,
                              Gtk.TreeViewColumn column)
  {
  	Gtk.TreeIter iter;
	  Gtk.TreeModel model;

    model = this.tree_view.get_model ();
	  return_if_fail (model != null);

	  model.get_iter (out iter, path);

	  this.toggle_active (iter, model);
  }

  private new bool button_press_event (Gtk.Widget sender, Gdk.EventButton event)
  {
    return false;
  }

  private new bool popup_menu (Gtk.Widget sender)
  {
    return true;
  }

  private void set_active (Gtk.TreeIter iter, Gtk.TreeModel model, bool active)
  {
    PluginRegistrar pr;
    Gtk.ListStore list_store;

    list_store = model as Gtk.ListStore;
    list_store.get (iter, Col.INFO, out pr);

    if (active)
    {
      /* activate the plugin */
      pr.activate ();
    }
    else
    {
      /* deactivate the plugin */
      pr.desactivate ();
    }
    list_store.set (iter, Col.ACTIVE, pr.is_active);
    ConfigManager.get_instance ().set_boolean ("Plugins", pr.name, active);
  }

  private void toggle_active (Gtk.TreeIter iter, Gtk.TreeModel model)
  {
    bool active;

    model.get (iter, Col.ACTIVE, out active);
    this.set_active (iter, model, !active);
  }

  private void populate_lists ()
  {
    Gtk.TreeIter iter;
    Gtk.ListStore model;

    model = this.tree_view.get_model () as Gtk.ListStore;
    model.clear ();

    foreach (PluginRegistrar pr in this.plugin_manager.plugins)
    {
      model.append (out iter);
      model.set (iter, Col.ACTIVE, pr.is_active,
                       Col.INFO, pr);
    }

    if (model.get_iter_first (out iter))
    {
      PluginRegistrar pr;
      Gtk.TreeSelection selection;

      selection = this.tree_view.get_selection ();
      return_if_fail (selection != null);

      selection.select_iter (iter);
      model.get (iter, Col.INFO, out pr);
      this.configure_button.set_sensitive (pr.is_configurable);
    }
  }

  private void about_activated ()
  {
    PluginRegistrar pr;

    pr = this.get_selected_plugin();
    if (pr != null)
    {
      Gtk.AboutDialog dialog;

      dialog = new Gtk.AboutDialog ();
      dialog.authors = pr.authors;
      dialog.comments = pr.description;
      dialog.program_name = pr.name;
      dialog.website = pr.website;
      dialog.copyright = pr.copyright;

      dialog.run ();
      dialog.destroy ();
    }
  }

  private void configure_activated ()
  {
    PluginRegistrar pr;

    pr = this.get_selected_plugin ();
    if (pr != null && pr.is_configurable)
    {
      pr.preferences ();
    }
  }

  /**
   * Create a new Valide.PluginPreferences
   */
  public PluginPreferences (PluginManager plugin_manager)
  {
    Object (plugin_manager: plugin_manager);
  }

  construct
  {
    /* create and pack tree view */
    Gtk.ListStore list_store;
    Gtk.HButtonBox button_box;
    Gtk.TreeViewColumn column;
    Gtk.ScrolledWindow scrolled;
    Gtk.CellRendererText cell_text;
    Gtk.CellRendererToggle cell_toggle;

    scrolled = new Gtk.ScrolledWindow (null, null);
    scrolled.set_policy (Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC);
    scrolled.set_shadow_type (Gtk.ShadowType.IN);
    this.pack_start (scrolled, true, true, 0);

    this.tree_view = new Gtk.TreeView ();
    this.tree_view.set_rules_hint (true);
    this.tree_view.set_headers_visible (true);
    this.tree_view.set_headers_clickable (true);
    scrolled.add (this.tree_view);

  	list_store = new Gtk.ListStore (Col.NB, typeof (bool),
  	                                        typeof (PluginRegistrar));
    this.tree_view.set_model (list_store);

    /* first column */
    cell_toggle = new Gtk.CellRendererToggle ();
    cell_toggle.toggled.connect (active_toggled);
    column = new Gtk.TreeViewColumn.with_attributes (_("Enable"), cell_toggle,
                                                     "active", Col.ACTIVE);
    this.tree_view.append_column (column);

    /* second column */
    column = new Gtk.TreeViewColumn ();
    column.set_title (_("Plugin"));
    column.set_resizable (true);

  	cell_text = new Gtk.CellRendererText ();
    column.pack_start (cell_text, true);
    cell_text.ellipsize = Pango.EllipsizeMode.END;
    column.set_cell_data_func (cell_text, info_cell);

    column.set_spacing (6);
    this.tree_view.append_column (column);

    /* Sort on the plugin names */
    list_store.set_default_sort_func (model_name_sort);
    /* -1 == GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID */
    list_store.set_sort_column_id (-1, Gtk.SortType.ASCENDING);

    /* Enable search for our non-string column */
    this.tree_view.set_search_column (Col.INFO);
    this.tree_view.set_search_equal_func (name_search);

    this.tree_view.cursor_changed.connect (this.cursor_changed);
    this.tree_view.row_activated.connect (this.row_activated);
    this.tree_view.button_press_event.connect (this.button_press_event);
    this.tree_view.popup_menu.connect (this.popup_menu);

    button_box = new Gtk.HButtonBox ();
    button_box.set_layout (Gtk.ButtonBoxStyle.END);
    this.about_button = new Gtk.Button.from_stock (Gtk.Stock.ABOUT);
    button_box.pack_start (this.about_button, false, false, 0);
    this.about_button.clicked.connect (about_activated);
    this.configure_button = new Gtk.Button.from_stock (Gtk.Stock.PREFERENCES);
    button_box.pack_start (this.configure_button, false, false, 0);
    this.configure_button.clicked.connect (configure_activated);

    this.pack_start (button_box, false, true, 0);

    if (plugin_manager.plugins  != null)
    {
	    this.populate_lists ();
    }
    else
    {
	    this.about_button.set_sensitive (false);
	    this.configure_button.set_sensitive (false);		
    }
  }
}

