/* plugin-registrar.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

public errordomain Valide.PluginError
{
  MODULE_LOADING,
  MODULE_VERSION
}

/**
 * This class represents a registered plugin
 */
public class Valide.PluginRegistrar : GFix.TypeModule
{
  private Type type;
  private Module module;
  private Plugin plugin;
  private string module_path;
  private PreferencePluginFunction preferences_plugin;

  /**
   * The file extension for the plugin information file
   */
  public const string[] FILE_EXT = {"valide-plugin"};

  private delegate Type RegisterPluginFunction (TypeModule module) throws PluginError;
  private delegate Gtk.Widget PreferencePluginFunction (Plugin self);

  private string module_name
  {
    get;
    private set;
  }

  private int vapi_version
  {
    get;
    private set;
  }

  /**
   * The main window of the application
   */
  public Window window
  {
    get;
    construct;
  }

  /**
   * The path of plugin information file
   */
  public string path
  {
    get;
    construct;
  }

  /**
   * Get if theplugin is active
   */
  public bool is_active
  {
    get
    {
      return (this.plugin != null);
    }
  }

  /**
   * Get if the plugin is activable
   */
  public bool is_available
  {
    get;
    private set;
  }

  /**
   * The plugin name
   */
  public string name
  {
    get;
    private set;
  }

  /**
   * The list of plugin authors
   */
  public string[] authors
  {
    get;
    private set;
  }

  /**
   * The plugin description
   */
  public string description
  {
    get;
    private set;
  }

  /**
   * The plugin icon file
   */
  public string icon_name
  {
    get;
    private set;
  }

  /**
   * The plugin website
   */
  public string website
  {
    get;
    private set;
  }

  /**
   * The plugin copyriht
   */
  public string copyright
  {
    get;
    private set;
  }

  /**
   * Get if the plugin is configurable
   */
  public bool is_configurable
  {
    get
    {
      return (this.preferences_plugin != null);
    }
  }

  private void register_plugin () throws Error
  {
    this.module_path = Module.build_path (Path.get_dirname (this.path),
                                          this.module_name);

    if (this.vapi_version <= int.parse (Config.VAPI_VERSION))
    {
      if (FileUtils.test (this.module_path, FileTest.EXISTS))
      {
        this.module = Module.open (this.module_path, ModuleFlags.BIND_LAZY);
        if (this.module != null)
        {
          void* function;
          RegisterPluginFunction register_plugin = null;

          this.module.symbol ("register_plugin", out function);
          if (function != null)
          {
            register_plugin = (RegisterPluginFunction)function;

            //try
            //{
              this.type = register_plugin ((TypeModule)this);

              this.module.symbol ("create_configure_dialog", out function);
              this.preferences_plugin = (PreferencePluginFunction)function;
            /*}
            catch (Error e)
            {
              this.is_available = false;
              throw new PluginError.MODULE_LOADING ("%s", e.message);
            }*/
          }
          else
          {
            throw new PluginError.MODULE_LOADING (_("invalid module"));
          }
        }
        else
        {
          throw new PluginError.MODULE_LOADING ("%s", module.error);
        }
      }
      else
      {
         throw new PluginError.MODULE_LOADING (_("the module doesn't exist"));
      }
    }
    else
    {
      throw new PluginError.MODULE_VERSION (_("Incompatible vapi version"));
    }
  }

  construct
  {
    assert (Module.supported ());
    this.preferences_plugin = null;
  }

  /**
   * Register a new plugin
   *
   * @param filename The path of plugin
   */
  public PluginRegistrar (string filename, Window window)
  {
    Object (path: filename, window: window);
  }

  /**
   * @see TypeModule.load
   */
  public override bool load ()
  {
    try
    {
      KeyFile key_file;

      key_file = new KeyFile ();
      key_file.load_from_file (this.path, KeyFileFlags.NONE);
      this.module_name = key_file.get_string ("Valide Plugin", "Module");
      this.vapi_version = key_file.get_integer ("Valide Plugin", "Vapi");
      this.name = key_file.get_string ("Valide Plugin", "Name");
      this.description = key_file.get_string ("Valide Plugin", "Description");
      this.icon_name = key_file.get_string ("Valide Plugin", "Icon");
      this.authors = key_file.get_string_list ("Valide Plugin", "Authors");
      this.website = key_file.get_string ("Valide Plugin", "Website");
      this.copyright = key_file.get_string ("Valide Plugin", "Copyright");
      this.is_available = true;

      this.register_plugin ();
      this.is_available = true;
    }
    catch (Error e)
    {
      debug (_("Can't load the plugin '%s': %s"), this.path, e.message);
      this.description = e.message;
      if (this.name == null)
      {
        this.name = Path.get_basename (Path.get_dirname (this.path));
      }
      this.is_available = false;
    }
    return this.is_available;
  }

  /**
   * @see TypeModule.unload
   */
  public override void unload ()
  {
    this.module = null;
  }

  /**
   * Activate the plugin
   */
  public void activate ()
  {
    if (this.is_available)
    {
      this.plugin = Object.new (this.type,
                                "path", Path.get_dirname (this.module_path),
                                "window", this.window) as Plugin;
    }
  }

  /**
   * Desactivate the plugin
   */
  public void desactivate ()
  {
    this.plugin = null;
  }

  /**
   * Show the plugin preferences
   */
  public void preferences ()
  {
    if (this.is_available && this.is_active && this.is_configurable)
    {
      Gtk.Dialog dialog;

      dialog = new Gtk.Dialog.with_buttons ("Plugin preferences",
                                            null,
                                            Gtk.DialogFlags.MODAL,
                                            Gtk.Stock.CLOSE,
                                            Gtk.ResponseType.CLOSE);
      dialog.vbox.pack_start (this.preferences_plugin (this.plugin),
                              true, true, 0);
      dialog.show_all ();
      dialog.run ();
      dialog.destroy ();
    }
  }
}

