/* preferences-dialog.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 * 	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/* @todo set_smart_home_end */
/**
 * The preferences dialog box
 */
public class Valide.PreferencesDialog : AbstractPreferencesDialog
{
  private enum Col
  {
    ID,
    TEXT,
    NB
  }

  /**
   * The main window of the application
   */
  public Window window
  {
    get;
    construct;
  }

  private void update_param (Gtk.Widget sender, string group)
  {
    Value val;
    string key;
    string widget_type;
    ConfigManager config;

    config = ConfigManager.get_instance ();
    key = sender.get_name ().replace ("_", "-");
    widget_type = sender.get_type ().name ();
    switch (widget_type)
    {
      case "GtkSpinButton":
        Gtk.SpinButton btn;

        btn = sender as Gtk.SpinButton;
        val = Value (typeof (int));
        val.set_int (btn.get_value_as_int ());
        config.set_integer (group, key, val.get_int ());
      break;
      case "GtkFontButton":
        Gtk.FontButton btn;

        btn = sender as Gtk.FontButton;
        val = Value (typeof (string));
        val.set_string (btn.get_font_name ());
        config.set_string (group, key, val.get_string ());
      break;
      case "GtkCheckButton":
        Gtk.CheckButton btn;

        btn = sender as Gtk.CheckButton;
        val = Value (typeof (bool));
        val.set_boolean (btn.get_active ());
        config.set_boolean (group, key, val.get_boolean ());
      break;
      case "GtkFileChooserButton":
        Gtk.FileChooserButton btn;

        btn = sender as Gtk.FileChooserButton;
        val = Value (typeof (string));
        val.set_string (btn.get_filename ());
        config.set_string (group, key, val.get_string ());
      break;
      case "GtkTreeView":
      case "GtkComboBox":
        string id;
        Gtk.TreeIter iter;
        Gtk.TreeModel tree_model;

        if (widget_type == "GtkTreeView")
        {
          Gtk.TreeView tree_view;

          tree_view = sender as Gtk.TreeView;
          tree_view.get_selection ().get_selected (out tree_model, out iter);
        }
        else
        {
          Gtk.ComboBox combo_box;

          combo_box = sender as Gtk.ComboBox;
          tree_model = combo_box.get_model ();
          combo_box.get_active_iter (out iter);
        }
        tree_model.get (iter, Col.ID, out id);
        val = Value (typeof (string));
        val.set_string (id);
        config.set_string (group, key, id);
      break;
      default:
        val = Value (typeof (string));
        val.set_string ("");
        debug ("Unknow type: %s", sender.get_type ().name ());
      break;
    }

    /* Update GtkSourceView display */
    switch (group)
    {
      case "GtkSourceView":
        foreach (Document document in this.window.documents.documents)
        {
          document.split_view.set_property (key, val);
        }
      break;
      case "Interface":
        this.window.set_property (key, val);
      break;
    }
  }

  /**
   * Callback for update the editor preferences
   */
  [CCode (instance_pos = -1)]
  protected void update_editor_param (Gtk.Widget sender)
  {
    this.update_param (sender, "GtkSourceView");
  }

  /**
   * Callback for update the interface preferences
   */
  [CCode (instance_pos = -1)]
  protected void update_interface_param (Gtk.Widget sender)
  {
    this.update_param (sender, "Interface");
  }

  /**
   * Callback for update the font preferences
   */
  [CCode (instance_pos = -1)]
  protected void update_font_param (Gtk.Widget sender)
  {
    this.update_param (sender, "GtkSourceView");
  }

  /**
   * Callback for update the project preferences
   */
  [CCode (instance_pos = -1)]
  protected void update_projects_param (Gtk.Widget sender)
  {
    this.update_param (sender, "Projects");
  }

  private void set_param () throws Error
  {
    KeyFile key_file;
    string[] groups;
    ConfigManager config;

    config = ConfigManager.get_instance ();

    key_file = new KeyFile ();
    key_file.load_from_file (config.get_config_file(), KeyFileFlags.NONE);

    groups = key_file.get_groups ();
    foreach (string g in groups)
    {
      try
      {
        string[] keys;

        keys = key_file.get_keys (g);
        foreach (string k in keys)
        {
          string name;
          Object object;

          name = k.replace ("-", "_");
          object = this.xml.get_object (name);

          if (object != null)
          {
            switch (object.get_type ().name ())
            {
              case "GtkSpinButton":
                Gtk.SpinButton btn;

                btn = object as Gtk.SpinButton;
                btn.set_value ((double)config.get_integer (g, k));
              break;
              case "GtkFontButton":
                Gtk.FontButton btn;

                btn = object as Gtk.FontButton;
                btn.set_font_name (config.get_string (g, k));
              break;
              case "GtkCheckButton":
                Gtk.CheckButton btn;

                btn = object as Gtk.CheckButton;
                btn.set_active (config.get_boolean (g, k));
              break;
              case "GtkFileChooserButton":
                Gtk.FileChooserButton btn;

                btn = object as Gtk.FileChooserButton;
                btn.select_filename (config.get_string (g, k));
              break;
              case "GtkTreeView":
                string id;
                string select;
                Gtk.TreeIter iter;
                Gtk.TreeView tree_view;
                Gtk.TreeModel tree_model;

                select = config.get_string (g, k);
                tree_view = object as Gtk.TreeView;
                tree_model = tree_view.get_model ();
                if (tree_model.get_iter_first (out iter))
                {
                  do
                  {
                    tree_model.get (iter, Col.ID, out id);
                    if (id == select)
                    {
                      tree_view.get_selection ().select_iter (iter);
                      break;
                    }
                  } while (tree_model.iter_next (ref iter));
                }
              break;
              case "GtkComboBox":
                string id;
                string select;
                Gtk.TreeIter iter;
                Gtk.ComboBox combo_box;
                Gtk.TreeModel tree_model;

                select = config.get_string (g, k);
                combo_box = object as Gtk.ComboBox;
                tree_model = combo_box.get_model ();
                if (tree_model.get_iter_first (out iter))
                {
                  do
                  {
                    tree_model.get (iter, Col.ID, out id);
                    if (id == select)
                    {
                      combo_box.set_active_iter (iter);
                      break;
                    }
                  } while (tree_model.iter_next (ref iter));
                }
              break;
              default:
                debug ("Unknow type: %s", object.get_type ().name ());
              break;
            }
          }
        }
      }
      catch (Error e)
      {
        debug (e.message);
      }
    }
  }

  private void populate_style_scheme ()
  {
    string text;
    Gtk.TreeIter iter;
    Gtk.TreeView tree_view;
    Gtk.ListStore list_store;
    Gtk.TreeViewColumn col;
    Gtk.CellRenderer renderer;
    Gtk.SourceStyleScheme scheme;
    Gtk.SourceStyleSchemeManager manager;

    list_store = new Gtk.ListStore (Col.NB, typeof (string), typeof (string));
    list_store.set_sort_column_id (Col.ID, Gtk.SortType.ASCENDING);

    tree_view = this.widgets.style_scheme;
    tree_view.set_model (list_store);
    tree_view.set_headers_visible (false);

    col = new Gtk.TreeViewColumn ();
    col.set_title (_("Style"));

    renderer = new Gtk.CellRendererText ();
    col.pack_start (renderer, false);
    col.set_attributes (renderer, "markup", Col.TEXT);

    tree_view.append_column (col);

    manager = new Gtk.SourceStyleSchemeManager ();
    foreach (string id in manager.get_scheme_ids ())
    {
      scheme = manager.get_scheme (id);
      text = "<b>%s</b> - %s".printf (scheme.name, scheme.description);
      list_store.append (out iter);
      list_store.set (iter, Col.ID, id, Col.TEXT, text);
    }
  }

  /**
   * Create a new Valide.PreferenceDialog
   *
   * @param window The main window of the application
   */
  public PreferencesDialog (Window window)
  {
    Object (window: window);
  }

  construct
  {
    Gtk.Widget layout;
    Gtk.CellRendererText cell;

    this.widgets.dialog.set_transient_for (this.window);

    this.widgets.show_right_margin.toggled.connect ((s) => {
      bool active;

      active = s.get_active ();
      this.widgets.label7.sensitive = active;
      this.widgets.right_margin_position.sensitive = active;
    });

    cell = new Gtk.CellRendererText ();
    this.widgets.switcher_style.pack_start (cell, true);
    this.widgets.switcher_style.set_attributes (cell, "text", Col.TEXT);

    cell = new Gtk.CellRendererText ();
    this.widgets.smart_home_end.pack_start (cell, true);
    this.widgets.smart_home_end.set_attributes (cell, "text", Col.ID);

    this.populate_style_scheme ();

    try
    {
      this.set_param ();
      this.widgets.show_right_margin.toggled ();
    }
    catch (Error e)
    {
      debug (e.message);
    }

    this.connect_signals ("valide_preferences_dialog_");

    this.widgets.notebook.append_page (this.window.plugins.preferences (),
                                       new Gtk.Label (_("Plugins")));
  }

  /**
   * @see Gtk.Dialog.run
   */
  public int run ()
  {
    this.widgets.dialog.show_all ();
    return this.widgets.dialog.run ();
  }

  /**
   * @see Gtk.Object.destroy
   */
  public override void destroy ()
  {
    this.widgets.dialog.destroy ();
    base.destroy ();
  }
}

