/* project-dialog-options.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 * 	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/**
 * A dialog box widget for show/edit the option of a project
 */
public class Valide.ProjectDialogOptions : AbstractProjectDialogOptions
{
  /**
   * The project
   */
  public Project project
  {
    get;
    construct;
  }

  /**
   * Create a new Valide.ProjectDialogOption
   *
   * @param project The attached project
   * @param parent The parent window
   */
  public ProjectDialogOptions (Project project, Gtk.Window? parent)
  {
    Object (project: project);
    this.widgets.dialog.set_transient_for (parent);
  }

  construct
  {
    string opt;
    Gdk.Pixbuf icon;
    Gtk.TreeIter iter;
    Gtk.IconView icon_view;
    Gtk.ListStore list_store;
    Gtk.CellRendererText cell;

    icon_view = this.widgets.icon_view;
    list_store = new Gtk.ListStore (3, typeof (Gdk.Pixbuf),
                                       typeof (string),
                                       typeof (int));
    icon_view.set_model (list_store);
    icon_view.set_pixbuf_column (0);
    icon_view.set_text_column (1);

    try
    {
      list_store.append (out iter);
      icon = new Gdk.Pixbuf.from_file (Path.build_filename (Config.PIXMAPS_DIR, "pref-general.png"));
      list_store.set (iter, 0, icon, 1, _("General"), 2, 0, -1);
    }
    catch (Error e)
    {
      debug (e.message);
    }

    try
    {
      Builder builder;

      list_store.append (out iter);
      icon = new Gdk.Pixbuf.from_file (Path.build_filename (Config.PIXMAPS_DIR, "pref-compiler.png"));
      list_store.set (iter, 0, icon, 1, _("Builder"), 2, 1, -1);
      builder = project.builders.create_builder (project);
      this.widgets.notebook.append_page (builder.preferences (), null);
    }
    catch (Error e)
    {
      debug (e.message);
    }

    try
    {
      list_store.append (out iter);
      icon = new Gdk.Pixbuf.from_file (Path.build_filename (Config.PIXMAPS_DIR, "pref-executable.png"));
      list_store.set (iter, 0, icon, 1, _("Executable"), 2, 2, -1);
      this.widgets.notebook.append_page (Executable.preferences (project), null);
    }
    catch (Error e)
    {
      debug (e.message);
    }

    icon_view.selection_changed.connect ((s) => {
      List<Gtk.TreePath> tabs;

      tabs = s.get_selected_items ();
      if (tabs != null)
      {
        int index;
        Gtk.TreeIter iter2;

        s.model.get_iter (out iter2, tabs.data);
        s.model.get (iter2, 2, out index, -1);
        this.widgets.notebook.set_current_page (index);
      }
    });
    /* Set first item */
    icon_view.select_path (new Gtk.TreePath.from_string ("0"));

    this.widgets.path.set_text (this.project.path);

    this.widgets.name.set_text (this.project.name);
    this.widgets.name.changed.connect ((s) => {
      this.project.name = (s as Gtk.Entry).get_text ();
      this.project.save ();
    });

    this.widgets.author.set_text (this.project.author);
    this.widgets.author.changed.connect ((s) => {
      this.project.author = (s as Gtk.Entry).get_text ();
      this.project.save ();
    });

    this.widgets.version.set_text (this.project.version);
    this.widgets.version.changed.connect ((s) => {
      this.project.version = (s as Gtk.Entry).get_text ();
      this.project.save ();
    });

    this.widgets.license.set_text (this.project.license);

    list_store = this.widgets.builder.model as Gtk.ListStore ;
    cell = new Gtk.CellRendererText ();
    this.widgets.builder.pack_start (cell, true);
    this.widgets.builder.set_attributes (cell, "text", 0);
    foreach (string builder in BuilderManager.get_instance ().get_builders ())
    {
      list_store.append (out iter);
      list_store.set (iter, 0, builder);
      if (builder.down () == this.project.builder.down ())
      {
        this.widgets.builder.set_active_iter (iter);
      }
    }
    this.widgets.builder.changed.connect ((s) => {
      Gtk.TreeIter it;

      if (s.get_active_iter (out it))
      {
        string b;
        Type type;
        Builder builder;

        s.model.get (it, 0, out b);
        type = Type.from_name ("Valide" + b + "BuilderOptions");
        if (type != 0)
        {
          try
          {
            this.project.builder = b;
            /* Replace options */
            this.project.builder_options = Object.new (type) as BuilderOptions;
            /* Replace the builder widget */
            builder = project.builders.create_builder (this.project);
            this.widgets.notebook.remove_page (1);
            this.widgets.notebook.insert_page (builder.preferences (), null, 1);
            this.widgets.notebook.show_all ();

            this.project.save ();
          }
          catch (Error e)
          {
            warning (e.message);
          }
        }
        else
        {
          warning (_("Coulnd't use this builder"));
        }
      }
    });

    this.connect_signals ("valide_project_dialog_options_");
  }

  /**
   * @see Gtk.Dialog.run
   */
  public int run ()
  {
    this.widgets.dialog.show_all ();
    return this.widgets.dialog.run ();
  }

  /**
   * @see Gtk.Object.destroy
   */
  public override void destroy ()
  {
    this.widgets.dialog.destroy ();
    base.destroy ();
  }
}

