/* recent-manager-ui.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/**
 * The recent files/projects manager
 */
public class Valide.RecentManagerUI : Object
{
  private uint id_file = 0;
  private uint id_project = 0;
  private Gtk.ActionGroup recents_action_group;

  /**
   * This signal is emitted when the recent files is updated
   */
  public signal void update_recent_files ();

  /**
   * This signal is emitted when the recent projects is updated
   */
  public signal void update_recent_projects ();

  /**
   * The ui manager
   */
  public UIManager ui_manager
  {
    get;
    construct;
  }

  /**
   * The document manager
   */
  public DocumentManager documents
  {
    get;
    construct;
  }

  /**
   * The project manager
   */
  public ProjectManager projects
  {
    get;
    construct;
  }

  /**
   * The config manager
   */
  public ConfigManager config
  {
    get;
    construct;
  }

  /**
   * The number of the recent files
   */
  public int max_recent_files
  {
    get
    {
      return this.config.get_integer ("Interface", "max-recent-files");
    }

    set
    {
      this.config.set_integer ("Interface", "max-recent-files", value);
      this.update_files ();
    }
  }

  /**
   * The number of the recent projects
   */
  public int max_recent_projects
  {
    get
    {
      return this.config.get_integer ("Interface", "max-recent-projects");
    }

    set
    {
      this.config.set_integer ("Interface", "max-recent-projects", value);
      this.update_projects ();
    }
  }

  private List<Gtk.RecentInfo> _recent_files = null;
  /**
   * The list of the recent files
   */
  public List<Gtk.RecentInfo> recent_files
  {
    get
    {
      int nb_items = 0;
      List<Gtk.RecentInfo> items;
      Gtk.RecentManager recent_manager;

      this._recent_files = null;
      recent_manager = Gtk.RecentManager.get_default ();
      items = recent_manager.get_items ();
      /* filter */
      foreach (Gtk.RecentInfo l in items)
      {
        if (nb_items >= this.max_recent_files)
        {
          break;
        }
        if (l.has_group ("valide") && FileUtils.test (l.get_uri (), FileTest.EXISTS)
            && (l.get_uri ().has_suffix (".vala")
                || l.get_uri ().has_suffix (".vapi")
                || l.get_uri ().has_suffix (".gs")))
        {
          this._recent_files.prepend (l);
          nb_items++;
        }
      }
      /* sort */
      this._recent_files.sort ((CompareFunc)this.sort_recents_mru);
      return this._recent_files;
    }
  }

  private List<Gtk.RecentInfo> _recent_projects = null;
  /**
   * The list of the recent projects
   */
  public List<Gtk.RecentInfo> recent_projects
  {
    get
    {
      int nb_items = 0;
      List<Gtk.RecentInfo> items;
      Gtk.RecentManager recent_manager;

      this._recent_projects = null;
      /* filter */
      recent_manager = Gtk.RecentManager.get_default ();
      items = recent_manager.get_items ();
      foreach (Gtk.RecentInfo l in items)
      {
        if (nb_items >= this.max_recent_projects)
        {
          break;
        }
        if (l.has_group ("valide") && FileUtils.test (l.get_uri (), FileTest.EXISTS)
            && l.get_uri ().has_suffix (".vide"))
        {
          this._recent_projects.prepend (l);
          nb_items++;
        }
      }
      /* sort */
      this._recent_projects.sort ((CompareFunc)this.sort_recents_mru);
      return this._recent_projects;
    }
  }

  private static int sort_recents_mru (Gtk.RecentInfo a, Gtk.RecentInfo b)
  {
    return (int)(b.get_modified () - a.get_modified ());
  }

  private void recents_menu_activate (Gtk.Action sender)
  {
    string uri;
    Gtk.RecentInfo info;

    info = sender.get_data<Gtk.RecentInfo> ("gtk-recent-info");
    uri = info.get_uri ();
    debug (uri);

    try
    {
      if (this.projects.accept_file (uri))
      {
        this.projects.open (uri);
      }
      else
      {
        this.documents.create (uri);
      }
    }
    catch (Error e)
    {
      warning (e.message);
    }
  }

  private void add (string uri, string mime_type)
  {
    Gtk.RecentData data;

    data = Gtk.RecentData ();
    data.display_name = uri;
    data.description = mime_type;
    data.mime_type = mime_type;
    data.app_name = "Val(a)IDE";
    data.app_exec = "valide";
    string[] groups = new string[2];
    groups[0] = "valide";
    groups[1] = null;
    data.groups = groups;
    Gtk.RecentManager.get_default ().add_full (uri, data);
  }

  private void update (UIManager ui_manager, List<Gtk.RecentInfo> items,
                       string ui_path, ref uint id)
  {
    List<Gtk.Action> actions;
    Gtk.RecentManager recent_manager;

    if (id != 0)
    {
      ui_manager.remove_ui (id);
    }

    this.recents_action_group = new Gtk.ActionGroup ("RecentActions");
    ui_manager.insert_action_group (this.recents_action_group, 0);

    actions = recents_action_group.list_actions ();
    foreach (Gtk.Action l in actions)
    {
      l.activate.disconnect (this.recents_menu_activate);
       recents_action_group.remove_action (l);
    }

    id = ui_manager.new_merge_id ();

    recent_manager = Gtk.RecentManager.get_default ();

    int i = 0;
    foreach (Gtk.RecentInfo info in items)
    {
      string ext;
      string uri;
      string tip;
      string label;
      string escaped;
      Gtk.Action action;
      string action_name;

      i++;
      uri = info.get_uri_display ();

      ext = Utils.get_extension (uri);
      action_name = "recent" + ext + "-%d".printf (i);
      escaped = Path.get_basename (info.get_display_name ());

      if (i >= 10)
      {
        label = "%d.  %s".printf (i, escaped);
      }
      else
      {
        label = "_%d.  %s".printf (i, escaped);
      }

      /* Translators: %s is a URI */
      tip = _("Open '%s'").printf (uri);

      action = new Gtk.Action (action_name, label, tip, null);
      action.set_data ("gtk-recent-info", info);

      action.activate.connect (recents_menu_activate);
      this.recents_action_group.add_action (action);

      ui_manager.add_ui (id, ui_path, action_name, action_name,
                         Gtk.UIManagerItemType.MENUITEM, false);
    }
  }

  /**
   * Create a new Valide.RecentManagerIU
   *
   * @param ui_manager The ui manager
   * @param documents The document manager
   * @param projects The projects manager
   */
  public RecentManagerUI (UIManager ui_manager, DocumentManager documents,
                          ProjectManager projects, ConfigManager config)
  {
    Object (ui_manager: ui_manager, documents: documents, projects: projects,
            config: config);
  }  

  construct
  {
    this.documents.tab_added.connect ((s, d) => {
      this.add_file (d.path);
    });
    this.projects.project_opened.connect ((s, p) => {
      this.add_project (p.filename);
    });
    this.update_files ();
    this.update_projects ();
  }

  /**
   * Add a file
   */
  public void add_file (string uri)
  {
    if (uri.has_suffix (".vala"))
    {
      this.add (uri, "vala");
    }
    if (uri.has_suffix (".vapi"))
    {
      this.add (uri, "vapi");
    }
    else if (uri.has_suffix (".gs"))
    {
      this.add (uri, "genie");
    }
    this.update_files ();
  }

  /**
   * Add a project
   */
  public void add_project (string uri)
  {
    this.add (uri, "vide");
    this.update_projects ();
  }

  /**
   * Update the recent files
   */
  public void update_files ()
  {
    this.update (this.ui_manager, this.recent_files,
                 "/menubar/file/file-recents-placeholder", ref this.id_file);
    this.update_recent_files ();
  }

  /**
   * Update the recent projects
   */
  public void update_projects ()
  {
    this.update (this.ui_manager, this.recent_projects,
                 "/menubar/project/project-recents-placeholder",
                 ref this.id_project);
    this.update_recent_projects ();
  }
}

