/* source-view.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

private enum ChangeCaseAction
{
  TO_UPPER_CASE,
  TO_LOWER_CASE,
}

/**
 * A widget for show highlighting text
 */
public class Valide.SourceView : Gtk.SourceView
{
  private GotoDialog goto_dialog = null;

  public signal void search ();
  public signal void find_next ();
  public signal void find_prev ();
  public signal void replace ();

  /**
   * The buffer
   */
  public SourceBuffer source_buffer
  {
    get
    {
      return this.buffer as SourceBuffer;
    }
  }

  /**
   * The text font
   */
  public string font
  {
    set
    {
      Pango.FontDescription desc;

      desc =  Pango.FontDescription.from_string (value);
      this.modify_font (desc);
    }
  }

  /**
   * The style scheme
   */
  public string style_scheme
  {
    set
    {
      Gtk.SourceStyleScheme scheme;
      Gtk.SourceStyleSchemeManager manager;

      manager = new Gtk.SourceStyleSchemeManager ();
      if (value != "")
      {
        scheme = manager.get_scheme (value);
      }
      else
      {
        scheme = manager.get_scheme ("classic");
      }
      this.source_buffer.set_style_scheme (scheme);
    }
  }

  public bool draw_spaces_space
  {
    get
    {
      return this._get_draw_spaces (Gtk.SourceDrawSpacesFlags.SPACE);
    }
    set
    {
      this._set_draw_spaces (Gtk.SourceDrawSpacesFlags.SPACE, value);
    }
  }

  public bool draw_spaces_tab
  {
    get
    {
      return this._get_draw_spaces (Gtk.SourceDrawSpacesFlags.TAB);
    }
    set
    {
      this._set_draw_spaces (Gtk.SourceDrawSpacesFlags.TAB, value);
    }
  }

  public bool draw_spaces_newline
  {
    get
    {
      return this._get_draw_spaces (Gtk.SourceDrawSpacesFlags.NEWLINE);
    }
    set
    {
      this._set_draw_spaces (Gtk.SourceDrawSpacesFlags.NEWLINE, value);
    }
  }

  public bool draw_spaces_nbsp
  {
    get
    {
      return this._get_draw_spaces (Gtk.SourceDrawSpacesFlags.NBSP);
    }
    set
    {
      this._set_draw_spaces (Gtk.SourceDrawSpacesFlags.NBSP, value);
    }
  }

  public bool draw_spaces_leading
  {
    get
    {
      return this._get_draw_spaces (Gtk.SourceDrawSpacesFlags.LEADING);
    }
    set
    {
      this._set_draw_spaces (Gtk.SourceDrawSpacesFlags.LEADING, value);
    }
  }

  public bool draw_spaces_text
  {
    get
    {
      return this._get_draw_spaces (Gtk.SourceDrawSpacesFlags.TEXT);
    }
    set
    {
      this._set_draw_spaces (Gtk.SourceDrawSpacesFlags.TEXT, value);
    }
  }

  public bool draw_spaces_trailing
  {
    get
    {
      return this._get_draw_spaces (Gtk.SourceDrawSpacesFlags.TRAILING);
    }
    set
    {
      this._set_draw_spaces (Gtk.SourceDrawSpacesFlags.TRAILING, value);
    }
  }

  /**
   * @see Gtk.SourceView.smart_home_end
   */
  public new string smart_home_end
  {
    get
    {
      switch (base.smart_home_end)
      {
        case Gtk.SourceSmartHomeEndType.BEFORE:
          return "before";
        case Gtk.SourceSmartHomeEndType.AFTER:
          return "after";
        case Gtk.SourceSmartHomeEndType.ALWAYS:
          return "always";
        case Gtk.SourceSmartHomeEndType.DISABLED:
        default:
          return "disabled";
      }
    }
    set
    {
      switch (value)
      {
        case "disabled":
          base.smart_home_end = Gtk.SourceSmartHomeEndType.DISABLED;
        break;
        case "before":
          base.smart_home_end = Gtk.SourceSmartHomeEndType.BEFORE;
        break;
        case "after":
          base.smart_home_end = Gtk.SourceSmartHomeEndType.AFTER;
        break;
        case "always":
          base.smart_home_end = Gtk.SourceSmartHomeEndType.ALWAYS;
        break;
      }
    }
  }

  private void _set_draw_spaces (Gtk.SourceDrawSpacesFlags flag, bool value)
  {
    if (value)
    {
      this.draw_spaces |= flag;
    }
    else
    {
      this.draw_spaces &= ~flag;
    }
  }

  private bool _get_draw_spaces (Gtk.SourceDrawSpacesFlags flag)
  {
    return (this.draw_spaces & flag) != 0;
  }

  private new void scroll_to_iter (Gtk.TextIter iter)
  {
    Gtk.TextMark mark;

    mark = this.buffer.create_mark ("cursor", iter, true);
    this.scroll_to_mark (mark, 0.4, false, 0, 0);
    this.buffer.delete_mark (mark);
  }

  [CCode (instance_pos = -1)]
  private void on_action_goto_line (SourceView sender)
  {
    this.goto_line ();
  }

  [CCode (instance_pos = -1)]
  private void on_action_search (SourceView sender)
  {
    this.search ();
  }

  [CCode (instance_pos = -1)]
  private void on_action_find_next (SourceView sender)
  {
    this.find_next ();
  }

  [CCode (instance_pos = -1)]
  private void on_action_find_prev (SourceView sender)
  {
    this.find_prev ();
  }

  [CCode (instance_pos = -1)]
  private void on_action_replace (SourceView sender)
  {
    this.replace ();
  }

  private void change_case (ChangeCaseAction action)
  {
    string text;
    Gtk.TextIter end;
    Gtk.TextIter start;

    this.buffer.begin_user_action ();

    this.buffer.get_selection_bounds (out start, out end);

    text = this.buffer.get_text (start, end, false);
    switch (action)
    {
      case ChangeCaseAction.TO_UPPER_CASE:
        text = text.up ();
      break;
      case ChangeCaseAction.TO_LOWER_CASE:
        text = text.down ();
      break;
    }
    this.buffer.delete (start, end);
    this.buffer.insert (start, text, -1);

    this.buffer.get_iter_at_offset (out end, start.get_offset () - (int)text.length);
    this.buffer.select_range (start, end);

    this.buffer.end_user_action ();
  }

  [CCode (instance_pos = -1)]
  private void on_action_lower (SourceView sender)
  {
    this.change_case (ChangeCaseAction.TO_LOWER_CASE);
  }

  [CCode (instance_pos = -1)]
  private void on_action_upper (SourceView sender)
  {
    this.change_case (ChangeCaseAction.TO_UPPER_CASE);
  }

  static construct
  {
    unowned Gtk.BindingSet binding_set;

    binding_set = Gtk.BindingSet.by_class (typeof (SourceView).class_ref ());
    action_new (typeof (SourceView), "action_goto_line");
    Gtk.BindingEntry.add_signal (binding_set, Gdk.KeySyms.i,
                                 Gdk.ModifierType.CONTROL_MASK,
                                 "action_goto_line", 0);

    action_new (typeof (SourceView), "action_search");
    Gtk.BindingEntry.add_signal (binding_set, Gdk.KeySyms.f,
                                 Gdk.ModifierType.CONTROL_MASK,
                                 "action_search", 0);

    action_new (typeof (SourceView), "action_find_next");
    Gtk.BindingEntry.add_signal (binding_set, Gdk.KeySyms.g,
                                 Gdk.ModifierType.CONTROL_MASK,
                                 "action_find_next", 0);

    action_new (typeof (SourceView), "action_find_prev");
    Gtk.BindingEntry.add_signal (binding_set, Gdk.KeySyms.g,
                                 Gdk.ModifierType.CONTROL_MASK
                                   | Gdk.ModifierType.SHIFT_MASK,
                                 "action_find_prev", 0);

    action_new (typeof (SourceView), "action_replace");
    Gtk.BindingEntry.add_signal (binding_set, Gdk.KeySyms.h,
                                 Gdk.ModifierType.CONTROL_MASK,
                                 "action_replace", 0);

    action_new (typeof (SourceView), "action_lower");
    Gtk.BindingEntry.add_signal (binding_set, Gdk.KeySyms.l,
                                 Gdk.ModifierType.CONTROL_MASK,
                                 "action_lower", 0);

    action_new (typeof (SourceView), "action_upper");
    Gtk.BindingEntry.add_signal (binding_set, Gdk.KeySyms.u,
                                 Gdk.ModifierType.CONTROL_MASK,
                                 "action_upper", 0);
  }

  construct
  {
    try
    {
      string[] keys;
      KeyFile key_file;
      ConfigManager config;

      this.buffer = new SourceBuffer ();
      config = ConfigManager.get_instance ();

      key_file = new KeyFile ();
      key_file.load_from_file (config.get_config_file(), KeyFileFlags.NONE);

      keys = key_file.get_keys ("GtkSourceView");
      foreach (string key in keys)
      {
        string val;
        string name;
        Value gvalue;

        name = key.replace ("-", "_");
        val = config.get_string ("GtkSourceView", key);
        if (val == "false" || val == "true")
        {
          gvalue = Value (typeof (bool));
          gvalue.set_boolean ((val == "true") ? true : false);
        }
        else
        {
          double dval;

          if (double.try_parse (val, out dval))
          {
            gvalue = Value (typeof (int));
            gvalue.set_int (int.parse (val));
          }
          else
          {
            gvalue = Value (typeof (string));
            gvalue.set_string (val);
          }
        }
        this.set_property (name, gvalue);
      }
    }
    catch (Error e)
    {
      debug (e.message);
    }

    Signal.connect (this, "action_goto_line",
                    (Callback)this.on_action_goto_line, this);
    Signal.connect (this, "action_search",
                    (Callback)this.on_action_search, this);
    Signal.connect (this, "action_find_next",
                    (Callback)this.on_action_find_next, this);
    Signal.connect (this, "action_find_prev",
                    (Callback)this.on_action_find_prev, this);
    Signal.connect (this, "action_replace",
                    (Callback)this.on_action_replace, this);
    Signal.connect (this, "action_lower",
                    (Callback)this.on_action_lower, this);
    Signal.connect (this, "action_upper",
                    (Callback)this.on_action_upper, this);
  }

  ~SourceView ()
  {
    /** @bug #674520 */
    this.buffer = null;
  }

  /**
   * Scroll text view to cursor
   */
  public void scroll_to_cursor ()
  {
    Gtk.TextBuffer buffer;

    buffer = this.buffer;
    this.scroll_to_mark (buffer.get_insert (), 0.25, false, 0.0, 0.0);
  }

  public bool goto_line (int line = -1, int col = 0)
  {
    bool ret = false;
    uint line_count;
    Gtk.TextIter iter;

    if (line < 0)
    {
      if (this.goto_dialog == null)
      {
        Gtk.Window toplevel;

        this.goto_dialog = new GotoDialog (this);
        toplevel = this.get_ancestor (typeof (Gtk.Window)) as Gtk.Window;
        if (toplevel != null)
        {
          Gtk.WindowGroup group;

          group = toplevel.group;
          if (group != null)
          {
            group.add_window (this.goto_dialog);
          }
        }
      }
      this.goto_dialog.show ();
    }
    else
    {
      line_count = this.buffer.get_line_count ();

      if (line <= line_count)
      {
        this.grab_focus ();
        this.buffer.get_iter_at_line_offset (out iter, line, col);
        this.buffer.place_cursor (iter);
        this.scroll_to_iter (iter);
        ret = true;
      }
    }
    return ret;
  }
}

