/* template.vala
 *
 * Copyright (C) 2008-2009  Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 * 	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/**
 * A class representing a template for project
 */
public class Valide.Template : Object, GLib.YAML.Buildable
{
  public string filename;

  private static Type[] child_types = {
     typeof (Source),
     typeof (Source),
     typeof (Source),
     typeof (Package),
     typeof (VapiDir)
  };
  private static const string[] child_tags = {
    "files",
    "files_vala",
    "files_genie",
    "packages",
    "vapi_dir"
  };

  /**
   * The generic file list of the template
   */
  public List<Source> files;

  /**
   * The file list of the template for the vala language
   */
  public List<Source> files_vala;

  /**
   * The file list of the template for the genie language
   */
  public List<Source> files_genie;

  /**
   * The list of the packages
   */
  public List<Package> packages;

  /**
   * The list of the vapi directory
   */
  public List<VapiDir> vapi_dir;

  /**
   * The name of the template
   */
  public string name { get; set; }

  /**
   * The description of the template
   */
  public string description { get; set; }

  /**
   * The path of the template file
   */
  public string path { get; private set; }

  /**
   * The icon of the template
   */
  public string icon { get; set; }

  /**
   * The builder
   */
  public string builder { get; set; }

  /**
   * The list of the compiler options
   */
  public BuilderOptions builder_options { get; set; }

  private Template ()
  {
  }

  /**
   * @see Glib.YAML.Buildable.add_child
   */
	public void add_child (GLib.YAML.Builder builder, Object child, string? type) throws Error
	{
	  if (type == "files")
	  {
		  this.files.prepend ((Source)child);
		}
	  if (type == "files_vala")
	  {
		  this.files_vala.prepend ((Source)child);
		}
	  if (type == "files_genie")
	  {
		  this.files_genie.prepend ((Source)child);
		}
		else if (type == "packages")
	  {
		  this.packages.prepend ((Package)child);
		}
		else if (type == "vapi_dir")
	  {
		  this.vapi_dir.prepend ((VapiDir)child);
		}
	}

  static construct
  {
    GLib.YAML.Buildable.register_type (typeof (Template), child_tags, child_types);
  }

  /**
   * Create a new template from file
   *
   * @param filename The template file
   */
  public static Template _new (string filename) throws Error
  {
    FileStream file;
    Template template;

    file = FileStream.open (filename, "r");
    if (file != null)
    {
      GLib.YAML.Builder builder;

      builder = new GLib.YAML.Builder ("Valide");
      template = builder.build_from_file (file) as Template;
      template.path = Path.get_dirname (filename);
    }
    else
    {
      throw new IOError.NOT_FOUND ("The file '%s' doesn't exist", filename); 
    }
    return template;
  }
}

