/* window.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

/**
 * The main window of the application
 */
public class Valide.Window : Gtk.Window
{
  public uint workspace;

  private Gdl.Dock dock;
  private uint status_context;
  private Gdl.DockLayout layout;
  private uint layout_menu_id = 0;
  private Gdk.WindowState window_state;
  private FullscreenToolbar fullscreen_toolbar;

  /**
   * Widget placement
   */
  public enum Placement
  {
    NONE = 0,
    TOP,
    BOTTOM,
    RIGHT,
    LEFT,
    CENTER,
    FLOATING
  }

  private delegate void ManagerActionFunction (Object manager) throws Error;

  /**
   * Lock the GDL dock items
   */
  public bool lock_items
  {
    get
    {
      return this.config_manager.get_boolean ("Interface", "lock-items");
    }
    set
    {
      this.config_manager.set_boolean ("Interface", "lock-items", value);
      this.layout.master.foreach ((o) => {
        if (o is Gdl.DockItem)
        {
          if (this.lock_items)
          {
            (o as Gdl.DockItem).lock ();
          }
          else
          {
            (o as Gdl.DockItem).unlock ();
          }
        }
      });
    }
  }

  private string _switcher_style;
  /**
   * The switcher style of the GDL notebook
   */
  public string switcher_style
  {
    get
    {
      this._switcher_style = this.config_manager.get_string ("Interface",
                                                           "switcher-style");
      if (this._switcher_style == "")
      {
        this._switcher_style = "both";
      }
      return this._switcher_style;
    }
    set
    {
      Gdl.SwitcherStyle style;

      this.config_manager.set_string ("Interface", "switcher-style", value);
      switch (value)
      {
        case "icon":
          style = Gdl.SwitcherStyle.ICON;
        break;
        case "text":
          style = Gdl.SwitcherStyle.TEXT;
        break;
        case "desktop":
          style = Gdl.SwitcherStyle.TOOLBAR;
        break;
        case "notebook":
          style = Gdl.SwitcherStyle.TABS;
        break;
        case "both":
        default:
          style = Gdl.SwitcherStyle.BOTH;
        break;
      }
      this.layout.master.switcher_style = style;
    }
  }

  /**
   * Number of recents files
   */
  public int max_recent_files
  {
    get
    {
      return this.recent_manager.max_recent_files;
    }

    set
    {
      this.recent_manager.max_recent_files = value;
    }
  }

  /**
   * Number of recents projects
   */
  public int max_recent_projects
  {
    get
    {
      return this.recent_manager.max_recent_projects;
    }

    set
    {
      this.recent_manager.max_recent_projects = value;
    }
  }

  /**
   * The UI manager
   */
  public UIManager ui_manager
  {
    get;
    private set;
  }

  /**
   * The recent file manager
   */
  public RecentManagerUI recent_manager
  {
    get;
    private set;
  }

  /**
   * The config manager
   */
  public ConfigManager config_manager
  {
    get;
    private set;
  }

  /**
   * The document manager
   */
  public DocumentManager documents
  {
    get;
    private set;
  }

  /**
   * The project manager
   */
  public ProjectManager projects
  {
    get;
    private set;
  }

  /**
   * The builder manager
   */
  public BuilderManager builders
  {
    get;
    private set;
  }

  /**
   * The executable manager
   */
  public ExecutableManager executables
  {
    get;
    private set;
  }

  /**
   * The plugin manager
   */
  public PluginManager plugins
  {
    get;
    private set;
  }

  /**
   * The menu bar
   */
  public Gtk.MenuBar menu
  {
    get;
    private set;
  }

  /**
   * The tool bar
   */
  public Gtk.Toolbar toolbar
  {
    get;
    private set;
  }

  /**
   * The statusbar
   */
  public Statusbar statusbar
  {
    get;
    private set;
  }

  private void message (string? log_domain, LogLevelFlags log_levels,
                        string message)
  {
    if (log_levels == LogLevelFlags.LEVEL_DEBUG)
    {
      if (Config.DEBUG)
      {
        print ("%s\n", message);
      }
    }
    else
    {
      Gtk.Dialog dialog;
      Gtk.MessageType type;

      switch (log_levels)
      {
        case LogLevelFlags.LEVEL_MESSAGE:
        case LogLevelFlags.LEVEL_INFO:
          type = Gtk.MessageType.INFO;
        break;
        case LogLevelFlags.LEVEL_WARNING:
          type = Gtk.MessageType.WARNING;
        break;
        case LogLevelFlags.LEVEL_CRITICAL:
        case LogLevelFlags.LEVEL_ERROR:
          type = Gtk.MessageType.ERROR;
        break;
        default:
          type = Gtk.MessageType.OTHER;
        break;
      }

      dialog = new Gtk.MessageDialog (this, Gtk.DialogFlags.MODAL, type,
                                      Gtk.ButtonsType.OK, message);
      dialog.run ();
      dialog.destroy ();
    }
  }

  private bool on_delete_event ()
  {
    int width;
    int height;
    bool propagate = true;

    this.get_size (out width, out height);
    this.config_manager.set_integer ("General", "width", width);
    this.config_manager.set_integer ("General", "height", height);
    this.layout.save_to_file (this.config_manager.get_layout_file ());
    try
    {
      this.executables.stop ();
    }
    catch (Error e)
    {
      debug (e.message);
    }

    if (this.documents.get_n_pages () > 0)
    {
      propagate = this.documents.close_all ();
    }
    return !propagate;
  }

  private void menu_item_select_cb (Gtk.Item item)
  {
    string message;
    Gtk.Action action;

    action = item.get_data<Gtk.Action> ("gtk-action");
    return_if_fail (action != null);

    action.get ("tooltip", out message);
    if (message != null)
    {
      this.statusbar.push (this.status_context, message);
    }
  }

  private void menu_item_deselect_cb (Gtk.Item item)
  {
    this.statusbar.pop (this.status_context);
  }

  private void connect_proxy_cb (Gtk.UIManager sender, Gtk.Action action,
                                 Gtk.Widget proxy)
  {
    if (proxy is Gtk.MenuItem)
    {
      Gtk.MenuItem item;

      item = proxy as Gtk.MenuItem;
      item.select.connect (this.menu_item_select_cb);
      item.deselect.connect (this.menu_item_deselect_cb);
    }
  }

  private void disconnect_proxy_cb (Gtk.UIManager sender, Gtk.Action action,
                                    Gtk.Widget proxy)
  {
    if (proxy is Gtk.MenuItem)
    {
      Gtk.MenuItem item;

      item = proxy as Gtk.MenuItem;
      item.select.disconnect (this.menu_item_select_cb);
      item.deselect.disconnect (this.menu_item_deselect_cb);
    }
  }

  private void update_layout_menu ()
  {
    string action_name;
    Gtk.ToggleAction action;
    List<Gdl.DockItem> items;
    Gtk.ActionGroup action_group;

    if (this.layout_menu_id != 0)
    {
      this.ui_manager.remove_ui (this.layout_menu_id);
    }

    action_group = new Gtk.ActionGroup ("LayoutActions");
    this.ui_manager.insert_action_group (action_group, 0);

    this.layout_menu_id = this.ui_manager.new_merge_id ();

    items = new List<Gdl.DockItem> ();
    this.layout.master.foreach ((o) => {
      if (o is Gdl.DockItem)
      {
        items.append (o as Gdl.DockItem);
      }
    });
    items.sort ((a, b) => {
      return Utils.strcmp ((a as Gdl.DockItem).long_name,
                           (b as Gdl.DockItem).long_name);
    });

    foreach (Gdl.DockItem item in items)
    {
      action_name = "view-layout." + item.name;
      action = new Gtk.ToggleAction (action_name, item.long_name,
                                     _("Show/Hide %s").printf (item.long_name),
                                     null);
      action.set_data ("dockitem", item);
      if (item.is_attached ())
      {
        action.set_active (true);
      }
      else
      {
        action.set_active (false);
      }
      action.toggled.connect ((a) => {
        Gdl.DockItem dock_item;

        dock_item = a.get_data<Gdl.DockItem> ("dockitem");
        if (a.active)
        {
          dock_item.show_item ();
        }
        else
        {
          dock_item.hide_item ();
        }
      });
      action_group.add_action (action);

      this.ui_manager.add_ui (this.layout_menu_id,
                              "/menubar/view/view-layout-placeholder",
                              action_name, action_name,
                              Gtk.UIManagerItemType.MENUITEM, false);
    }
  }

  private void setup_ui_manager ()
  {
    this.ui_manager = new UIManager ();
    this.ui_manager.connect_proxy.connect (this.connect_proxy_cb);
    this.ui_manager.disconnect_proxy.connect (this.disconnect_proxy_cb);
    this.ui_manager.populate ();

    this.menu = this.ui_manager.get_widget ("/menubar") as Gtk.MenuBar;
    this.toolbar = this.ui_manager.get_widget ("/toolbar") as Gtk.Toolbar;

    this.ui_manager.action_set_toggled ("window-show-statusbar", true);
    this.ui_manager.action_set_toggled ("window-show-toolbar", true);

    this.add_accel_group (this.ui_manager.get_accel_group ());
  }

  private void populate_window ()
  {
    Gtk.HBox hbox;
    Gtk.VBox main_box;
    Gdl.DockBar dockbar;

    main_box = new Gtk.VBox (false, 0);
    this.add (main_box);

    main_box.pack_start (this.menu, false, true, 0);
    main_box.pack_start (this.toolbar, false, true, 0);
    this.fullscreen_toolbar = new FullscreenToolbar (this);

    hbox = new Gtk.HBox (false, 0);
    main_box.pack_start (hbox, true, true, 0);

    dockbar = new Gdl.DockBar (dock);
    dockbar.set_style (Gdl.DockBarStyle.BOTH);
    hbox.pack_start (dockbar, false, true, 0);

    hbox.pack_start (this.dock, true, true, 0);

    try
    {
      Utils.register_icon (Path.build_filename (Config.PIXMAPS_DIR, "icone-16.xpm"),
                           "valide-project-icon");
    }
    catch (Error e)
    {
      debug (e.message);
    }
    this.add_widget (this.projects, "projects", _("Project"), Placement.LEFT,
                     "valide-project-icon");
    this.add_widget (this.documents, "editor", _("Editor"), Placement.CENTER,
                     Gtk.Stock.EDIT);
    this.add_widget (this.executables, "executables", _("Messages"),
                     Placement.BOTTOM, Gtk.Stock.EXECUTE);

    this.layout.load_layout ("__default__");
    main_box.pack_start (this.statusbar, false, true, 0);
  }

  private void on_action_activated (Gtk.UIManager sender, Gtk.Action action)
  {
    Object manager;
    string[] tokens;
    string manager_prefix;

    switch (action.name)
    {
      case "edit-redo":
        // @bug No key binding found
//        Gtk.bindings_activate (this.get_focus (), Gdk.KeySyms.z, Gdk.ModifierType.CONTROL_MASK | Gdk.ModifierType.SHIFT_MASK);
        Signal.emit_by_name (this.get_focus (), "redo");
        return;
      case "search-find-prev":
        // @bug No key binding found
//        Gtk.bindings_activate (this.get_focus (), Gdk.KeySyms.g, Gdk.ModifierType.CONTROL_MASK | Gdk.ModifierType.SHIFT_MASK);
        Signal.emit_by_name (this.get_focus (), "action_find_prev");
        return;
      case "config-preferences":
        this.config_manager.preferences (this);
        return;
    }

    tokens = action.name.split ("-", 2);
    if (tokens.length == 2)
    {
      Module module;

      switch (tokens[0])
      {
        case "config":
          manager = this.config_manager;
          manager_prefix = "valide_abstract_config_manager_";
        break;
        case "document":
          manager = this.documents;
          manager_prefix = "valide_document_manager_";
        break;
        case "executable":
          manager = this.executables;
          manager_prefix = "valide_executable_manager_";
        break;
        case "project":
          manager = this.projects;
          manager_prefix = "valide_project_manager_";
        break;
        case "window":
        default:
          manager = this;
          manager_prefix = "valide_window_";
        break;
      }

      module = Module.open (null, ModuleFlags.BIND_LAZY);
      if (module != null)
      {
        string method;
        void* function;

        if (action is Gtk.ToggleAction)
        {
          if (!(action as Gtk.ToggleAction).get_active ())
          {
            tokens[1] = "un" + tokens[1];
          }
        }
        method = manager_prefix + tokens[1].replace ("-", "_");
        module.symbol (method, out function);
        if (function != null)
        {
          ManagerActionFunction manager_action;

          manager_action = (ManagerActionFunction)function;
          try
          {
            manager_action (manager);
          }
          catch (Error e)
          {
            warning (e.message);
          }
        }
        else
        {
          string accel_path;

          accel_path = action.get_accel_path ();
          if (accel_path != null)
          {
            Fix.AccelKey key;

            if (Fix.AccelMap.lookup_entry (accel_path, out key))
            {
              Gtk.bindings_activate (this.get_focus (), key.accel_key,
                                     key.accel_mods);
            }
            else
            {
              debug ("Unknow action %s", method);
            }
          }
        }
      }
      else
      {
        error (Module.error ());
      }
    }
  }

  private void update_title ()
  {
    string title = "";
    Document document;
    Project project;

    document = this.documents.current;
    if (document != null)
    {
      if (document.path != "")
      {
        string filename;

        filename = Path.get_basename (document.path);
        if (document.is_new)
        {
          title += "%s - ".printf (filename);
        }
        else
        {
          string dir;

          dir = Utils.replace_home_dir_with_tilde (Path.get_dirname (document.path));
          title += "%s (%s) - ".printf (filename, dir);
        }
      }
    }

    project = this.projects.current;
    if (project != null)
    {
      if (project.name != "")
      {
        title += project.name + " - ";
      }
    }

    title += "Val(a)IDE";
    this.set_title (title);
  }

  private bool is_fullscreen ()
  {
    return ((this.window_state & Gdk.WindowState.FULLSCREEN) != 0);
  }

  construct
  {
    Log.set_handler (null, LogLevelFlags.LEVEL_MASK, this.message);

    this.get_group ().add_window (this);

    this.delete_event.connect (this.on_delete_event);

    this.window_state_event.connect ((s, e) => {
      this.window_state = e.new_window_state;
      if ((e.changed_mask &
          (Gdk.WindowState.MAXIMIZED | Gdk.WindowState.FULLSCREEN)) != 0)
      {
        bool full_screen;

        full_screen = ((e.new_window_state &
          (Gdk.WindowState.MAXIMIZED | Gdk.WindowState.FULLSCREEN)) != 0);

        this.statusbar.set_has_resize_grip (!full_screen);
        this.config_manager.set_boolean ("General", "full-screen", full_screen);
      }
    });

    this.setup_ui_manager ();

    this.dock = new Gdl.Dock ();

    this.config_manager = ConfigManager.get_instance ();

    this.layout = new Gdl.DockLayout (this.dock);
    this.layout.load_from_file (this.config_manager.get_layout_file ());
    this.switcher_style = this.config_manager.get_string ("Interface",
                                                          "switcher-style");
    this.update_layout_menu ();
    this.layout.master.layout_changed.connect (() => {
      this.update_layout_menu ();
    });

    this.documents = new DocumentManager (this.ui_manager);
    this.documents.tab_changed.connect (this.update_title);
    this.documents.tab_state_changed.connect (this.update_title);
    this.documents.tab_added.connect (this.update_title);
    this.documents.tab_added.connect (() => {
      this.present_widget (this.documents);
    });
    this.documents.tab_removed.connect (this.update_title);
    this.documents.tab_cursor_moved.connect ((s, d, r, c) => {
      this.statusbar.set_cursor_position (r, c);
    });

    this.executables = new ExecutableManager (this.ui_manager);
    this.executables.run_exec.connect (() => {
      this.present_widget (this.executables);
    });

    this.builders = BuilderManager.get_instance ();
    this.builders.executables = this.executables;

    this.projects = new ProjectManager (this.ui_manager, this.documents,
                                        this.builders);
    this.projects.project_options_changed.connect (() => {
      this.update_title ();
    });
    this.projects.project_opened.connect (this.update_title);
    this.projects.project_opened.connect (() => {
      this.present_widget (this.projects);
    });
    this.projects.project_closed.connect (this.update_title);

    this.recent_manager = new RecentManagerUI (this.ui_manager, this.documents,
                                               this.projects, this.config_manager);

    this.statusbar = new Statusbar ();
    this.status_context = this.statusbar.get_context_id ("valide_window");

    this.ui_manager.post_activate.connect (this.on_action_activated);
    this.plugins = new PluginManager (this);
    this.populate_window ();
    this.update_title ();

    try
    {
      List<Gdk.Pixbuf> list;

      list.append (new Gdk.Pixbuf.from_file (Path.build_filename (Config.PIXMAPS_DIR, "icone-16.xpm")));
      list.append (new Gdk.Pixbuf.from_file (Path.build_filename (Config.PIXMAPS_DIR, "icone-32.xpm")));
      list.append (new Gdk.Pixbuf.from_file (Path.build_filename (Config.PIXMAPS_DIR, "icone-48.xpm")));
      list.append (new Gdk.Pixbuf.from_file (Path.build_filename (Config.PIXMAPS_DIR, "icone-64.xpm")));
      Gtk.Window.set_default_icon_list (list);
    }
    catch (Error e)
    {
      debug (e.message);
    }
  }

  /**
   * Launch the application
   */
  public void run ()
  {
    bool full_screen;

    full_screen = this.config_manager.get_boolean ("General", "full-screen");
    if (full_screen)
    {
      this.maximize ();
    }
    else
    {
      int width;
      int height;

      width = this.config_manager.get_integer ("General", "width");
      height = this.config_manager.get_integer ("General", "height");
      if (width > 1 && height > 1)
      {
        this.resize (width, height);
      }
    }

    this.fullscreen_toolbar.hide ();
    this.show_all ();
  }

  public void show_toolbar ()
  {
    this.toolbar.show ();
  }

  public void unshow_toolbar ()
  {
    this.toolbar.hide ();
  }

  public void show_statusbar ()
  {
    this.statusbar.show ();
  }

  public void unshow_statusbar ()
  {
    this.statusbar.hide ();
  }

  /**
   * @see Gtk.Window.fullscreen
   */
  public new void fullscreen ()
  {
    if (!this.is_fullscreen ())
    {
      base.fullscreen ();
      this.menu.hide ();
      this.toolbar.hide ();
      this.statusbar.hide ();
      this.fullscreen_toolbar.show_all ();
    }
  }

  /**
   * @see Gtk.Window.unfullscreen
   */
  public new void unfullscreen ()
  {
    if (this.is_fullscreen ())
    {
      base.unfullscreen ();
      this.menu.show ();
      this.toolbar.show ();
      this.statusbar.show ();
      this.fullscreen_toolbar.hide ();
    }
  }

  public void leave_fullscreen ()
  {
    this.unfullscreen ();
    this.ui_manager.action_set_toggled ("window-fullscreen", false);
  }

  /**
   * Add a new widget to the window
   *
   * @param widget A widget
   * @param name Unique name for identifying the dock object
   * @param long_name Human readable name for the dock object
   * @param placement The position to dock item
   * @param stock_id Stock icon for the dock object
   */
  public void add_widget (Gtk.Widget widget, string name, string long_name,
                          Placement placement, string? stock_id = null)
  {
    Gdl.DockItem item;

    item = new Gdl.DockItem.with_stock (name, long_name, stock_id,
                                        Gdl.DockItemBehavior.NORMAL);
    item.add (widget);
    widget.set_data ("dockitem", item);
    this.dock.add_item (item, (Gdl.DockPlacement)placement);
    if (this.lock_items)
    {
      item.lock ();
    }
    item.show_all ();
  }

  /**
   * Remove a widget to the window added before with add_widget.
   *
   * @param widget A widget
   */
  public void remove_widget (Gtk.Widget widget)
  {
    Gdl.DockItem item;

    item = widget.get_data<Gdl.DockItem> ("dockitem");
    return_if_fail (item != null);

    item.remove (widget);
    item.unbind ();
  }

  /**
   * Make sure the widget is visible to user. If the widget is hidden, it will
   * be shown. If it is not visible to user, it will be made visible.
   *
   * @param widget A widget
   */
  public void present_widget (Gtk.Widget widget)
  {
    Gtk.Widget parent;
    Gdl.DockItem item;

    item = widget.get_data<Gdl.DockItem> ("dockitem");
    return_if_fail (item != null);

    /* Hack to present the dock item if it's in a notebook dock item */
    parent = item.get_parent ();
    if (parent is Gtk.Notebook)
    {
      int pagenum;
      Gtk.Notebook notebook;

      notebook = parent as Gtk.Notebook;
      pagenum = notebook.page_num (item);
      notebook.set_current_page (pagenum);
    }
    else if (!item.is_attached ())
    {
      item.show_item ();
    }

    /* FIXME: If the item is floating, present the window */
    /* FIXME: There is no way to detect if a widget was floating before it was
    detached since it no longer has a parent there is no way to access the
    floating property of the GdlDock structure.*/
  }

  /**
   * Close the window
   */
  public void quit ()
  {
    if (!this.on_delete_event ())
    {
      this.destroy ();
    }
  }

  public void about ()
  {
    AboutDialog dialog;

    dialog = new AboutDialog ();
    dialog.run ();
    dialog.destroy ();
  }
}
