/* advanced-search.vala
 *
 * Copyright (C) 2010-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

using Valide;

private class FindItem
{
  public string file;
  public string line;
  public string text;
}

public class AdvancedSearch : Plugin, Object
{
  enum Col
  {
    TEXT,
    FILE,
    LINE,
    NB
  }

  private static string grep_path;

  private bool stop;
  private uint ui_id;
  private Gtk.HBox box;
  private Gtk.Button stop_button;
  private Gtk.TreeView tree_view;
  private AdvancedSearchDialog dialog;
  private Gtk.ActionGroup action_group = null;

  const Gtk.ActionEntry[] entries = {
    { "search-advanced", null, "Advanced search", "<ctrl><shift>F",
      "Advanced search", on_search }
  };

  const string ui = """
      <ui>
        <menubar name="menubar">
          <menu action="search">
            <placeholder name="search-ops-1">
              <menuitem action="search-advanced"/>
            </placeholder>
          </menu>
        </menubar>
      </ui>""";

  /**
   * @see Plugin.path
   */
  public string path { get; construct set; }
  /**
   * @see Plugin.window
   */
  public Window window { get; construct set; }

  private void on_row_activated (Gtk.TreeView sender, Gtk.TreePath path,
                                 Gtk.TreeViewColumn column)
  {
    string line;
    string file;
    Gtk.TreeIter iter;
    Gtk.TreeModel model;

    model = this.tree_view.get_model ();
    model.get_iter (out iter, path);
    model.get (iter, Col.LINE, out line, Col.FILE, out file);

    if (file != "")
    {
      try
      {
        Document document;

        document = this.window.documents.create (file);
        document.text_view.goto_line (int.parse (line) - 1);
        document.grab_focus ();
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
  }

  private void add_line (FindItem item)
  {
    Gtk.TreeIter iter;
    Gtk.ListStore list_store = this.tree_view.get_model () as Gtk.ListStore;

    list_store.append (out iter);
    list_store.set (iter, Col.TEXT, item.text, Col.FILE, item.file,
                          Col.LINE, item.line);
  }

  private FindItem parse_line (string line)
  {
    FindItem item;

    item = new FindItem ();
    if (line.index_of_char (-1, ':') != 0)
    {
      string[] col;

      col = line.split (":", 3);
      item.file = col[0];
      item.line = col[1];
      item.text = col[2];
    }
    else
    {
      item.file = "";
      item.line = "";
      item.text = line;
    }
    return item;
  }

  /**
   * Execute the grep command and return stdout.
   */
  private string exec_grep (string filename, string pattern,
                            string[] options) throws Error
  {
    string output;
    StringBuilder command;

    /* build grep command */
    command = new StringBuilder ();
    command.append_printf ("%s ", this.grep_path);
    command.append (string.joinv (" ", options));
    command.append_printf (" -- '%s' '%s'", pattern.replace ("'", "\\'"),
                                            filename.replace ("'", "\\'"));

    /* execute command */
    Process.spawn_command_line_sync (command.str, out output, null, null);

    return output;
  }

  private bool populate_asc ()
  {
    string[] options = {};
    string[] filenames = {};

    if (!this.dialog.case_sensitive)
    {
      options += "-i";
    }
    if (!this.dialog.entire_word)
    {
      options += "-w";
    }
    if (!this.dialog.use_regex)
    {
      options += "-F";
    }
    options += "-nH";

    if (this.dialog.current_scope)
    {
      filenames += this.window.documents.current.path;
    }
    else if (this.dialog.opened_scope)
    {
      foreach (Document document in this.window.documents.documents)
      {
        filenames += document.path;
      }
    }
    else
    {
      foreach (Valide.Source source in this.window.projects.current.files)
      {
        filenames += Path.build_filename (this.window.projects.current.path, source.path);
      }
    }

    if (filenames.length > 0)
    {
      this.clear ();
      this.stop_button.sensitive = true;
      foreach (string filename in filenames)
      {
        string output;
        string[] lines;

        output = this.exec_grep (filename, this.dialog.pattern, options);
        lines = output.split ("\n");
        for (int i = 0; lines[i] != null; i++)
        {
          if (lines[i] != "")
          {
            FindItem item;

            item = this.parse_line (lines[i]);
            this.add_line (item);
          }
          Utils.process_gtk_events ();
        }
        if (this.stop)
        {
          this.stop = false;
          break;
        }
      }
      this.stop_button.sensitive = false;
    }
    else
    {
      warning (_("Invalid option!"));
    }
    return false;
  }

  private void on_search ()
  {
    this.dialog.active_current_scope = (this.window.documents.current != null);
    this.dialog.active_opened_scope = (this.window.documents.get_n_pages () > 1);
    this.dialog.active_project_scope = (this.window.projects.current != null);

    this.dialog.show_all ();
    if (this.dialog.run () == Gtk.ResponseType.ACCEPT)
    {
      if (this.dialog.pattern != "")
      {
        this.window.present_widget (this.box);
        Idle.add (this.populate_asc);
      }
      else
      {
        warning (_("Empty search string!"));
      }
    }
    this.dialog.hide ();
  }

  private void clear ()
  {
    Gtk.ListStore list_store;

    list_store = this.tree_view.get_model () as Gtk.ListStore;
    list_store.clear ();
  }

  private void setup_ui ()
  {
    bool active = false;

    if (this.action_group == null)
    {
      Gtk.UIManager ui_manager;

      ui_manager = this.window.ui_manager;
      this.action_group = new Gtk.ActionGroup ("advanced-search");
      this.action_group.add_actions (this.entries, this);
      ui_manager.insert_action_group (this.action_group, 0);
      try
      {
        this.ui_id = ui_manager.add_ui_from_string (ui, -1);
      }
      catch (Error e)
      {
        debug (e.message);
      }
    }

    if (this.window.documents.current != null
        || this.window.projects.current != null)
    {
      active = true;
    }
    this.action_group.get_action ("search-advanced").sensitive = active;
    this.tree_view.sensitive = active;
  }

  construct
  {
    Gtk.VBox vbox;
    Gtk.Button button;
    Gtk.UIManager ui_manager;
    Gtk.ListStore list_store;
    Gtk.TreeViewColumn column;
    Gtk.CellRendererText render;
    Gtk.ScrolledWindow scrolled_window;

    this.box = new Gtk.HBox (false, 0);

    vbox = new Gtk.VBox (false, 0);
    this.box.pack_start (vbox, false, false, 0);

    button = new Gtk.Button ();
    button.relief = Gtk.ReliefStyle.NONE;
    button.image = new Gtk.Image.from_stock (Gtk.Stock.CLEAR,
                                             Gtk.IconSize.SMALL_TOOLBAR);
    button.set_tooltip_text (_("Clear"));
    button.clicked.connect (this.clear);
    vbox.pack_start (button, false, false, 0);

    vbox.pack_start (new Gtk.HSeparator (), false, false, 0);

    this.stop = false;
    this.stop_button = new Gtk.Button ();
    this.stop_button.relief = Gtk.ReliefStyle.NONE;
    this.stop_button.image = new Gtk.Image.from_stock (Gtk.Stock.STOP,
                                                       Gtk.IconSize.SMALL_TOOLBAR);
    this.stop_button.set_tooltip_text (_("Stop"));
    this.stop_button.clicked.connect (() => {
      this.stop = true;
    });
    this.stop_button.sensitive = false;
    vbox.pack_start (this.stop_button, false, false, 0);

    scrolled_window = new Gtk.ScrolledWindow (null, null);
    scrolled_window.set_policy (Gtk.PolicyType.AUTOMATIC,
                                Gtk.PolicyType.AUTOMATIC);
    this.box.pack_start (scrolled_window, true, true, 0);

    list_store = new Gtk.ListStore (Col.NB, typeof (string), typeof (string),
                                    typeof (string));
    this.tree_view = new Gtk.TreeView.with_model (list_store);
    this.tree_view.row_activated.connect (this.on_row_activated);
    scrolled_window.add (this.tree_view);

    /* Text */
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("Text"), render, "text",
                                                     Col.TEXT);
    this.tree_view.append_column (column);
    /* File */
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("File"), render, "text",
                                                     Col.FILE);
    this.tree_view.append_column (column);
    /* Line */
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("Line"), render, "text",
                                                     Col.LINE);
    this.tree_view.append_column (column);

    this.window.add_widget (this.box, "advanced-search-plugin",
                            _("Advanced search"), Window.Placement.BOTTOM,
                            Gtk.Stock.FIND_AND_REPLACE);
    this.window.documents.tab_removed.connect (this.setup_ui);
    this.window.documents.tab_changed.connect (this.setup_ui);
    this.window.projects.project_opened.connect (this.setup_ui);
    this.window.projects.project_closed.connect (this.setup_ui);

    this.dialog = new AdvancedSearchDialog ();
    this.dialog.set_transient_for (this.window);
    this.dialog.modal = true;
    this.setup_ui ();
  }

  public static void init () throws PluginError
  {
    AdvancedSearch.grep_path = Environment.find_program_in_path ("grep");
    if (AdvancedSearch.grep_path == null)
    {
      throw new PluginError.MODULE_LOADING (_("You should install grep to activate this plugin!"));
    }
  }

  ~AdvancedSearch ()
  {
    this.window.ui_manager.remove_ui (this.ui_id);
    this.window.remove_widget (this.box);
  }
}

public Type register_plugin (TypeModule module, void* target) throws PluginError
{
  AdvancedSearch.init ();
  return typeof (AdvancedSearch);
}

