/* file-browser.vala
 *
 * Copyright (C) 2009-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 * 	Nicolas Joseph <nicolas.joseph@valaide.org>
 */

using Valide;

public class FileBrowser : Plugin, Object
{
  /* enum for tree model columns, not all is currently used */
  enum Col
  {
    ICON,
    NAME,
    PATH,
    NB
  }

  private string _cwd;
  private Gtk.VBox box;
  private bool _show_hidden;
  private Gtk.TreeView tree_view;
  private Gtk.FileChooserButton file_chooser;

  public string path { get; construct set; }
  public Window window { get; construct set; }

  private string cwd
  {
    get
    {
      return this._cwd;
    }
    set
    {
      this._cwd = value;
      this.file_chooser.set_filename (value);
      this.populate (this);
    }
  }

  private bool show_hidden
  {
    get
    {
      return this._show_hidden;
    }
    set
    {
      this._show_hidden = value;
      this.populate (this);
    }
  }

  private void populate_asc (Gtk.TreeIter? root = null)
  {
    Gtk.TreeStore model;

    model = this.tree_view.get_model () as Gtk.TreeStore;
    if (root == null)
    {
      model.clear ();
    }
    try
    {
      Dir dir;
      string path;
      string dirname;
      Gtk.TreeIter iter;
      Gdk.Pixbuf pixbuf;
      Gtk.TreeIter parent;
      unowned string filename;

      if (root == null)
      {
        dirname = this.cwd;
      }
      else
      {
        Gtk.TreeIter child;

        model.get (root, Col.PATH, out dirname);
        if (model.iter_children (out child, root))
        {
          string name;

          model.get (child, Col.NAME, out name);
          if (name == "...")
          {
            model.remove (child);
          }
          else
          {
            return;
          }
        }
      }

      dir = Dir.open (dirname);
      while ((filename = dir.read_name ()) != null)
      {
        try
        {
          File file;
          FileInfo file_info;

          path = Path.build_filename (dirname, filename);
          file = File.new_for_path (path);
          file_info = file.query_info (FILE_ATTRIBUTE_STANDARD_IS_HIDDEN, 
                                       FileQueryInfoFlags.NONE,
                                       null);

          if (!file_info.get_is_hidden () || this.show_hidden)
          {
            pixbuf = Utils.get_pixbuf_for_file (path, Gtk.IconSize.BUTTON);
            model.append (out parent, root);
            model.set (parent, Col.ICON, pixbuf, Col.NAME, filename, Col.PATH, path);
            if (FileUtils.test (path, FileTest.IS_DIR))
            {
              model.append (out iter, parent);
              model.set (iter, Col.NAME, "...");
            }
          }
        }
        catch (Error e)
        {
          debug (e.message);
        }
      }
    }
    catch (Error e)
    {
      warning (e.message);
    }
  }

  /**
   * Call to re-populate tree view.
   */
  private void populate (Object sender)
  {
    Idle.add (() => {
      this.populate_asc ();
      return false;
    });
  }

  private void on_row_activated (Gtk.TreeView sender, Gtk.TreePath tree_path,
                                 Gtk.TreeViewColumn column)
  {
    string path;
    Gtk.TreeIter iter;
    Gtk.TreeModel model;

    model = this.tree_view.get_model ();
    model.get_iter (out iter, tree_path);
    model.get (iter, Col.PATH, out path);

    if (FileUtils.test (path, FileTest.IS_DIR))
    {
      this.cwd = path;
      this.populate (this);
    }
    else
    {
      try
      {
        if (this.window.projects.accept_file (path))
        {
          this.window.projects.open (path);
        }
        else
        {
          this.window.documents.create (path);
        }
      }
      catch (Error e)
      {
        warning (e.message);
      }
    }
  }

  private int sort_func (Gtk.TreeModel model, Gtk.TreeIter a, Gtk.TreeIter b)
  {
    string path_a, path_b;

    model.get (a, Col.PATH, out path_a);
    model.get (b, Col.PATH, out path_b);

    return Utils.cmp_filename (path_a, path_b);
  }

  construct
  {
    Gtk.HBox tablabel;
    Gtk.Button button;
    Gtk.HBox button_box;
    Gtk.TreeStore store;
    Gtk.TreeViewColumn col;
    Gtk.CellRenderer renderer;
    Gtk.ScrolledWindow scrolled;
    Gtk.ToggleButton toggle_button;

    this.box = new Gtk.VBox (false, 0);

    /* create some nice padding */
    this.box.set_spacing (2);
    this.box.set_border_width (2);

    button_box = new Gtk.HBox (false, 2);
    this.box.pack_start (button_box, false, false, 0);
 
    button = new Gtk.Button ();
    button.relief = Gtk.ReliefStyle.NONE;
    button.image = new Gtk.Image.from_stock (Gtk.Stock.GO_UP,
                                             Gtk.IconSize.SMALL_TOOLBAR);
    button.set_tooltip_text (_("Goto parent folder"));
    button.clicked.connect (() => {
      this.cwd = Path.get_dirname (this.cwd);
    });
    button_box.pack_start (button, false, false, 0);
 
    button = new Gtk.Button ();
    button.relief = Gtk.ReliefStyle.NONE;
    button.image = new Gtk.Image.from_stock (Gtk.Stock.REFRESH,
                                             Gtk.IconSize.SMALL_TOOLBAR);
    button.set_tooltip_text (_("Refresh"));
    button.clicked.connect (() => {
      this.populate (this);
    });
    button_box.pack_start (button, false, false, 0);
 
    toggle_button = new Gtk.ToggleButton ();
    toggle_button.relief = Gtk.ReliefStyle.NONE;
    toggle_button.image = new Gtk.Image.from_stock (Gtk.Stock.FILE,
                                                    Gtk.IconSize.SMALL_TOOLBAR);
    toggle_button.set_tooltip_text (_("Show the hidden files"));
    toggle_button.toggled.connect ((s) => {
      this.show_hidden = s.get_active ();
    });
    button_box.pack_start (toggle_button, false, false, 0);

    this.file_chooser = new Gtk.FileChooserButton ("Select folder",
                                                   Gtk.FileChooserAction.SELECT_FOLDER);
    this.file_chooser.selection_changed.connect ((s) => {
      /* Don't use the property for don't launch an infinite loop */
      this._cwd = s.get_filename ();
      this.populate (s);
    });
    this.box.pack_start (this.file_chooser, false, false, 0);
 
    /* create and pack tree view */
    scrolled = new Gtk.ScrolledWindow (null, null);
    scrolled.set_policy (Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC);
    scrolled.set_shadow_type (Gtk.ShadowType.IN);
    this.box.pack_start (scrolled, true, true, 0);

    this.tree_view = new Gtk.TreeView ();
    this.tree_view.set_rules_hint (false);
    this.tree_view.set_headers_visible (false);
    scrolled.add (this.tree_view);

    /* create model */
    store = new Gtk.TreeStore (Col.NB, typeof (Gdk.Pixbuf),
                               typeof (string), typeof (string));
    store.set_sort_column_id (Col.NAME, Gtk.SortType.ASCENDING);
    store.set_sort_func (Col.NAME, this.sort_func);
    /* main column */
    col = new Gtk.TreeViewColumn ();
    col.set_title(_("File"));

    renderer = new Gtk.CellRendererPixbuf ();
    col.pack_start (renderer, false);
    col.set_attributes (renderer, "pixbuf", Col.ICON);

    renderer = new Gtk.CellRendererText ();
    col.pack_start (renderer, false);
    col.set_attributes (renderer, "markup", Col.NAME);

    this.tree_view.append_column (col);
    this.tree_view.set_model (store);

    this.cwd = Environment.get_home_dir ();

    /* attach signal callbacks for tree view */
    this.tree_view.row_activated.connect (on_row_activated);
    this.tree_view.row_expanded.connect ((s, i, p) => {
      this.populate_asc (i);
      this.tree_view.expand_row (p, false);
    });

    this.window.add_widget (this.box, "file-browser-plugin", _("Files"),
                            Window.Placement.LEFT, Gtk.Stock.OPEN);
  }

  ~FileBrowser ()
  {
    this.window.remove_widget (this.box);
  }
}

public Type register_plugin (TypeModule module)
{
  return typeof (FileBrowser);
}

