/* make-builder.vala
 *
 * Copyright (C) 2008-2010 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

using Valide;

/**
 * Builder based on make: http://www.gnu.org/software/make/
 */
public class MakeBuilder : Plugin, Builder, Object
{
  /**
   * @see Valide.Builder.project
   */
  public Project project { get; construct set; }

  /**
   * @see Valide.Builder.executables
   */
  public ExecutableManager executables { get; construct set; }

  /**
   * @see Valide.Plugin.path
   */
  public string path { get; construct set; }

  /**
   * @see Valide.Plugin.window
   */
  public Window window { get; construct set; }

  private int run_make (BuildingStep step)
  {
    MakeExecutable make;
    MakeBuilderOptions options;
    ExecutableOptions executable_options;

    make = new MakeExecutable (this.project);
    options = this.project.builder_options as MakeBuilderOptions;
    executable_options = new ExecutableOptions ();
    executable_options.working_dir = this.project.path;
    executable_options.arguments = options.to_string (step);
    return executables.run (make, executable_options);
  }

  public MakeBuilder (ExecutableManager executables, Project project)
  {
    Object (project: project, executables: executables);
  }

  construct
  {
    BuilderManager.get_instance ().register_builder (this.get_type (),
                                                     typeof (MakeBuilderOptions));
  }

  /**
   * @see Valide.Builder.configure
   */
  public int configure ()
  {
    MakeBuilderOptions options;
    ConfigureExecutable configure;
    ExecutableOptions executable_options;

    configure = new ConfigureExecutable ();
    options = this.project.builder_options as MakeBuilderOptions;
    executable_options = new ExecutableOptions ();
    executable_options.working_dir = this.project.path;
    executable_options.arguments = options.to_string (BuildingStep.CONFIGURE);
    return executables.run (configure, executable_options);
  }

  /**
   * @see Valide.Builder.build
   */
  public int build ()
  {
    return this.run_make (BuildingStep.BUILD);
  }

  /**
   * @see Valide.Builder.install
   */
  public int install ()
  {
    return this.run_make (BuildingStep.INSTALL);
  }

  /**
   * @see Valide.Builder.dist
   */
  public int dist ()
  {
    return this.run_make (BuildingStep.DIST);
  }

  /**
   * @see Valide.Builder.clean
   */
  public int clean ()
  {
    return this.run_make (BuildingStep.CLEAN);
  }

  /**
   * @see Valide.Builder.distclean
   */
  public int distclean ()
  {
    return this.run_make (BuildingStep.DISTCLEAN);
  }

  /**
   * @see Valide.Builder.uninstall
   */
  public int uninstall ()
  {
    return this.run_make (BuildingStep.UNINSTALL);
  }

  /**
   * @see Valide.Builder.preferences
   */
  public Gtk.Widget preferences ()
  {
    return new MakeBuilderPreferences (this.project);
  }
}

public Type register_plugin (TypeModule module)
{
  return typeof (MakeBuilder);
}

