/* todo-manager.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *   Nicolas Joseph <nicolas.joseph@valaide.org>
 */

using Valide;

private class TodoItem
{
  public string type;
  public string author;
  public string priority;
  public string text;
  public string file;
  public int line;
  public unowned Document document;
}

public class Todo : Plugin, Object
{
  enum Col
  {
    TYPE,
    TEXT,
    AUTHOR,
    PRIORITY,
    LINE,
    FILE,
    DOCUMENT,
    NB
  }

  private Gtk.TreeView tree_view;
  private DocumentManager documents;
  private Gtk.ScrolledWindow scrolled_window;

  /**
   * @see Plugin.path
   */
  public string path { get; construct set; }
  /**
   * @see Plugin.window
   */
  public Window window { get; construct set; }

  private bool only_load_active_tab
  {
    get
    {
      return this.window.config_manager.get_boolean ("Todo", "only-load-active-tab");
    }
    set
    {
      this.window.config_manager.set_boolean ("Todo", "only-load-active-tab", value);
      this.populate_list ();
    }
  }

  private void on_row_activated (Gtk.TreeView sender, Gtk.TreePath path,
                                 Gtk.TreeViewColumn column)
  {
    int line;
    Document document;
    Gtk.TreeIter iter;
    Gtk.TreeModel model;

    model = this.tree_view.get_model ();
    model.get_iter (out iter, path);
    model.get (iter, Col.LINE, out line, Col.DOCUMENT, out document, -1);

    this.documents.current = document;
    document.text_view.goto_line (line - 1);
    document.grab_focus ();
  }

  private void populate_list ()
  {
    Gtk.ListStore model;

    this.tree_view.sensitive = false;

    /* clear data */
    model = this.tree_view.get_model () as Gtk.ListStore;
    model.clear ();

    if (this.documents.current != null)
    {
      int nb_doc = 0;
      int nb_task = 0;
      List<Document> _doc = null;
      unowned List<Document> doc = null;

      /* find out if we want to load all symbols or just active tab */
      if (this.only_load_active_tab)
      {
        _doc = new List<Document> ();
        _doc.prepend (this.documents.current);
        doc = _doc;
      }
      else
      {
        doc = this.documents.documents;
      }

      foreach (Document document in doc)
      {
        List<TodoItem> items;

        nb_doc++;
        items = this.parse (document);
        if (items != null)
        {
          foreach (TodoItem item in items)
          {
            this.add_line (item);
            nb_task++;
          }
        }
      }
      if (nb_doc > 0)
      {
        this.tree_view.sensitive = true;
        if (nb_task > 0)
        {
          this.tree_view.set_tooltip_text (null);
        }
        else
        {
          string help_text = _("Mark todo with `@(TODO|FIXME|CHANGED|NOTE|OPTIMIZE|IMPROVE|BUG) (author#priority#): to do'");
          this.tree_view.set_tooltip_text (help_text);
        }
      }
      else
      {
        this.tree_view.sensitive = false;
      }
    }
  }

  private void on_tab_changed ()
  {
    this.populate_list ();
  }

  private void add_line (TodoItem item)
  {
    Gtk.TreeIter iter;
    Gtk.ListStore list_store;

    list_store = this.tree_view.get_model () as Gtk.ListStore;
    list_store.append (out iter);
    list_store.set (iter, Col.TYPE, item.type, Col.AUTHOR, item.author,
                          Col.PRIORITY, item.priority, Col.TEXT, item.text,
                          Col.LINE, item.line, Col.FILE, item.file,
                          Col.DOCUMENT, item.document);
  }

  private List<TodoItem> parse (Document d)
  {
    string text;
    string[] lines;
    Gtk.TextBuffer buffer;
    Gtk.TextIter start, end;
    List<TodoItem> items = null;

    buffer = d.buffer;
    buffer.get_start_iter (out start);
    buffer.get_end_iter (out end);
    text = buffer.get_text (start, end, false);
    lines = text.split ("\n");

    try
    {
      Regex regex;

      regex = new Regex ("@(TODO|FIXME|CHANGED|NOTE|OPTIMIZE|IMPROVE|BUG)\\s*(\\(([^#]*?)?(#(.*)#)?\\))?[:]?\\s*(.*)",
                         RegexCompileFlags.CASELESS);
      for (int i = 0; lines[i] != null; i++)
      {
        try
        {
          MatchInfo match_info;

          if (regex.match_full (lines[i], -1, 0, 0, out match_info))
          {
            while (match_info.matches ())
            {
              TodoItem item;

              item = new TodoItem ();
              item.type = match_info.fetch (1).up ();
              item.author = match_info.fetch (3);
              item.priority = match_info.fetch (5);
              item.text = match_info.fetch (6);
              if (item.text.has_suffix ("*/"))
              {
                item.text = item.text.substring (0, item.text.length - 2);
              }
              item.document = d;
              item.file = d.filename;
              item.line = i + 1;
              items.append ((owned)item);
              match_info.next ();
            }
          }
        }
        catch (Error e)
        {
          warning (e.message);
        }
      }
    }
    catch (Error e)
    {
      debug (e.message);
    }
    return items;
  }

  construct
  {
    Gtk.ListStore list_store;
    Gtk.TreeViewColumn column;
    Gtk.CellRendererText render;

    this.scrolled_window = new Gtk.ScrolledWindow (null, null);

    this.scrolled_window.set_policy (Gtk.PolicyType.AUTOMATIC,
                                     Gtk.PolicyType.AUTOMATIC);

    list_store = new Gtk.ListStore (Col.NB,
                                    typeof (string), typeof (string),
                                    typeof (string), typeof (string),
                                    typeof (int), typeof (string),
                                    typeof (Object));
    this.tree_view = new Gtk.TreeView.with_model (list_store);
    this.tree_view.row_activated.connect (this.on_row_activated);
    this.scrolled_window.add (this.tree_view);

    /* type */
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("Type"), render, "text", Col.TYPE, null);
    this.tree_view.append_column (column);
    /* Author */
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("Author"), render, "text", Col.AUTHOR, null);
    this.tree_view.append_column (column);
    /* Priority */
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("Priority"), render, "text", Col.PRIORITY, null);
    this.tree_view.append_column (column);
    /* Text */
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("Text"), render, "text", Col.TEXT, null);
    this.tree_view.append_column (column);
    /* Line */
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("Line"), render, "text", Col.LINE, null);
    this.tree_view.append_column (column);
    /* File */
    render = new Gtk.CellRendererText ();
    column = new Gtk.TreeViewColumn.with_attributes (_("File"), render, "text", Col.FILE, null);
    this.tree_view.append_column (column);

    this.documents  = this.window.documents;

    this.documents.tab_removed.connect (this.populate_list);
    this.documents.tab_changed.connect (this.on_tab_changed);
    this.documents.tab_saved.connect (this.populate_list);

    this.tree_view.sensitive = false;

    try
    {
      Utils.register_icon (Path.build_filename (Config.PIXMAPS_DIR, "plugins",
                                                "todo.png"),
                           "todo-plugin-icon");
    }
    catch (Error e)
    {
      debug (e.message);
    }
    this.window.add_widget (this.scrolled_window, "todo-plugin", _("Todo"),
                            Window.Placement.BOTTOM, "todo-plugin-icon");
  }

  ~Todo ()
  {
    this.documents.tab_removed.disconnect (this.populate_list);
    this.documents.tab_changed.disconnect (this.on_tab_changed);
    this.documents.tab_saved.disconnect (this.populate_list);
    this.tree_view.row_activated.disconnect (this.on_row_activated);

    this.window.remove_widget (this.scrolled_window);
  }

  public Gtk.Widget create_configure_dialog ()
  {
    Gtk.VBox vbox;
    Gtk.CheckButton check;

    vbox = new Gtk.VBox (true, 0);

    check = new Gtk.CheckButton.with_label (_("Only load symbols for active tab"));
    check.set_active (this.only_load_active_tab);
    check.toggled.connect ((s) => {
      this.only_load_active_tab = s.get_active ();
    });
    vbox.pack_start (check, true, true, 0);

    return vbox;
  }
}

public Type register_plugin (TypeModule module)
{
  return typeof (Todo);
}

public Gtk.Widget create_configure_dialog (Todo self)
{
  return self.create_configure_dialog ();
}

