/* waf-executable.vala
 *
 * Copyright (C) 2008-2011 Nicolas Joseph
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author:
 *  Nicolas Joseph <nicolas.joseph@valaide.org>
 */

using Valide;

public class WafExecutable : Compiler
{
  public BuildingStep step;

  private static const string EXECUTABLE_PATH = "python ./waf -pp";
  public override string executable
  {
    get { return EXECUTABLE_PATH; }
    set {}
  }

  public uint nb_task { get; set; }
  public uint current_task { get; set; }

  protected override string get_file (string line) throws Error
  {
    string filename;

    filename = Path.build_filename (this.project.path, base.get_file (line));
    if (!FileUtils.test (filename, FileTest.EXISTS))
    {
      unowned WafBuilderOptions options;

      options = this.project.builder_options as WafBuilderOptions;
      filename = Path.build_filename (options.build_dir, filename);
    }
    return filename;
  }

  private Gdk.Color parse_build_line (ref string line)
  {
    Gdk.Color color;

    Gdk.Color.parse (Utils.Color.DEFAULT, out color);
    if (line.has_prefix ("|"))
    {
      string[] tmp;
      string new_line = line;

      foreach (string item in line.split ("|"))
      {
        tmp = item.split (" ");
        if (tmp[0] == "Total")
        {
          this.nb_task = int.parse (tmp[1]);
        }
        else if (tmp[0] == "Current")
        {
          this.current_task = int.parse (tmp[1]);
        }
        else if (tmp[0] == "Inputs")
        {
          new_line = tmp[1];
        }
        else if (tmp[0] == "Outputs")
        {
          new_line += " → " + tmp[1];
        }
      }
      line = new_line;
    }
    else if (line.has_prefix ("Waf:"))
    {
      Gdk.Color.parse (Utils.Color.INFO, out color);
    }
    else if (line.index_of ("warning:") != -1)
    {
      Gdk.Color.parse (Utils.Color.WARNING, out color);
    }
    else if (line.index_of ("error:") != -1)
    {
      Gdk.Color.parse (Utils.Color.ERROR, out color);
    }
    return color;
  }

  /**
   * @see Valide.Executable.get_color_line
   */
  public override void get_color_line (ref string line, out Gdk.Color color)
  {
    base.get_color_line (ref line, out color);

    if (line.has_suffix ("finished successfully"))
    {
      Gdk.Color.parse (Utils.Color.SUCCESS, out color);
    }
    else
    {
      switch (this.step)
      {
        case BuildingStep.CONFIGURE:
          if (line.index_of (": ok") != -1 || line.index_of (": yes ") != -1)
          {
            Gdk.Color.parse (Utils.Color.SUCCESS, out color);
          }
          else if (line.index_of (": fail ") != -1
            || line.index_of (": not found ") != -1)
          {
            Gdk.Color.parse (Utils.Color.WARNING, out color);
          }
          else if (line.index_of ("error:") != -1)
          {
            Gdk.Color.parse (Utils.Color.ERROR, out color);
          }
        break;
        case BuildingStep.BUILD:
        case BuildingStep.INSTALL:
        case BuildingStep.DIST:
          color = this.parse_build_line (ref line);
        break;
      }
    }
  }

  /**
   * Create a new WafExecutable
   *
   * @param project A project
   */
  public WafExecutable (Project project)
  {
    Object (project: project);
  }
}

